/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.projects;

import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.wall.WallContentManager;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

import com.google.common.collect.Iterables;

/**
 * {@link ActivityType} implementation for content
 */
public class WebContentActivityType extends AbstractProjectsActivityType
{
    /** data name for the page id */
    public static final String PAGE_ID = "pageId";
    /** data name for the content type */
    public static final String CONTENT_TYPE = "contentType";
    /** data name for the content summary */
    public static final String CONTENT_SUMMARY = "contentSummary";
    /** data name for the content title */
    public static final String CONTENT_TITLE = "contentTitle";
    /** data name for the content id */
    public static final String CONTENT_ID = "contentId";
    
    private WallContentManager _wallContentManager;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _wallContentManager = (WallContentManager) serviceManager.lookup(WallContentManager.ROLE);
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        Content content = (Content) parameters.get(ObservationConstants.ARGS_CONTENT);
        activity.setValue(CONTENT_ID, content.getId());
        activity.setValue(CONTENT_TITLE, content.getTitle());
        // FIXME the getExcerpt method lose the line break
        activity.setValue(CONTENT_SUMMARY, _wallContentManager.getExcerpt(content, 150));
        activity.setValue(CONTENT_TYPE, content.getTypes()[0]);
        
        if (content instanceof WebContent)
        {
            Page page = Iterables.getFirst(((WebContent) content).getReferencingPages(), null);
            if (page != null)
            {
                activity.setValue(PAGE_ID, page.getId());
            }
        }
    }
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        return false;
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        WebContent content = (WebContent) args.get(ObservationConstants.ARGS_CONTENT);
        // Find project
        List<Project> projects = _projectManager.getProjectsForSite(content.getSite());
        return projects.isEmpty() ? null : projects.get(0);
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        Expression titleExpr = new StringExpression(CONTENT_TITLE, Operator.WD, pattern);
        Expression summaryExpr = new StringExpression(CONTENT_SUMMARY, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
        return new OrExpression(titleExpr, summaryExpr);
    }
}
