/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.calendars.events;

import java.time.ZonedDateTime;
import java.time.temporal.ChronoUnit;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.ametys.core.util.DateUtils;

/**
 * Occurrence of a calendar event
 */
public class CalendarEventOccurrence
{
    private ZonedDateTime _startDate;
    private ZonedDateTime _endDate;
    private boolean _isFullDay;
    private List<String> _resourceIds;
    private CalendarEvent _event;

    /**
     * Creates a {@link CalendarEventOccurrence}.
     * @param calendarEvent the calendar event
     * @param occurrenceDate the occurrence date
     */
    public CalendarEventOccurrence(CalendarEvent calendarEvent, ZonedDateTime occurrenceDate)
    {
        this._startDate = occurrenceDate;

        long diffInSeconds = ChronoUnit.SECONDS.between(calendarEvent.getStartDate(), calendarEvent.getEndDate());
        this._endDate = occurrenceDate.plusSeconds(diffInSeconds);
        this._isFullDay = calendarEvent.getFullDay();
        this._resourceIds = calendarEvent.getResources();
        this._event = calendarEvent;
    }

    /**
     * Get the date of the begin of the occurrence
     * @return the date of the occurrence
     */
    public ZonedDateTime getStartDate()
    {
        return _startDate;
    }

    /**
     * Set the startDate of the occurrence
     * @param startDate the start date
     */
    public void setStartDate(ZonedDateTime startDate)
    {
        this._startDate = startDate;
    }

    /**
     * Get the date of the end of the occurrence
     * @return the date of the occurrence
     */
    public ZonedDateTime getEndDate()
    {
        return _endDate;
    }

    /**
     * Set the endDate of the occurrence
     * @param endDate the end date
     */
    public void setEndDate(ZonedDateTime endDate)
    {
        this._endDate = endDate;
    }

    /**
     * Get if the occurrence last all the day
     * @return true if the occurrence last all the day
     */
    public boolean isFullDay()
    {
        return _isFullDay;
    }

    /**
     * Set if the occurrence last all the day
     * @param isFullDay true if the occurrence last all the day
     */
    public void setFullDay(boolean isFullDay)
    {
        this._isFullDay = isFullDay;
    }

    /**
     * Retrieves the resource id's list.
     * @return the resource id's list.
     */
    public List<String> getResourceIds()
    {
        return _resourceIds;
    }

    /**
     * Set the resources ids
     * @param resourceIds The list of ressource ids
     */
    public void setResourceIds(List<String> resourceIds)
    {
        this._resourceIds = resourceIds;
    }

    /**
     * Check if the occurrence is before a given date
     * @param date the date
     * @return true if the occurrence is before a given date
     */
    public boolean before(ZonedDateTime date)
    {
        return this.getStartDate().isBefore(date);
    }

    /**
     * Check if the occurrence is after a given date
     * @param date the date
     * @return true if the occurrence is after a given date
     */
    public boolean after(ZonedDateTime date)
    {
        return this.getStartDate().isAfter(date);
    }
    
    /**
     * Get the event
     * @return the event
     */
    public CalendarEvent getEvent()
    {
        return _event;
    }
    
    
    /**
     * occurrence info as a JSONizable map
     * @return the map
     */
    public Map<String, Object> toJSON()
    {
        Map<String, Object> result = new HashMap<>();
    
        String occurrenceDateIso = DateUtils.zonedDateTimeToString(_startDate);
        result.put("id", _event.getId() + "$" + occurrenceDateIso);
        result.put("occurrenceDate", occurrenceDateIso);
        
        ZonedDateTime startDateEvent = _startDate;
        ZonedDateTime endDateEvent = _endDate;
        
        if (_isFullDay)
        {
            result.put("endDateNextDay", DateUtils.zonedDateTimeToString(endDateEvent.plusDays(1)));
        }
        
        result.put("startDate", DateUtils.zonedDateTimeToString(startDateEvent));
        result.put("endDate", DateUtils.zonedDateTimeToString(endDateEvent));
        
        return result;
    }
}
