/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.categories;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.tag.AbstractTagProvider;
import org.ametys.cms.tag.CMSTag;
import org.ametys.cms.tag.TagTargetType;
import org.ametys.cms.tag.TagTargetTypeExtensionPoint;

/**
 * Class representing categories as CMS tag provider.
 * This directly use {@link CategoryJCRProvider} and transform the categories in {@link CMSTag}
 */
public class CategoryTagProvider extends AbstractTagProvider<CMSTag> implements Serviceable
{
    /** Target types */
    protected TagTargetTypeExtensionPoint _targetTypeEP;
    
    /** Category tag provider */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    
    /** The JCR category provider */
    protected CategoryJCRProvider _jcrCategoryProvider;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _targetTypeEP = (TagTargetTypeExtensionPoint) smanager.lookup(TagTargetTypeExtensionPoint.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _jcrCategoryProvider = (CategoryJCRProvider) _categoryProviderEP.getExtension(CategoryJCRProvider.class.getName());
    }
    
    public Map<String, CMSTag> getTags(Map<String, Object> contextualParameters)
    {
        Map<String, CMSTag> cmsTags = new HashMap<>();
        
        Map<String, Category> categories = _jcrCategoryProvider.getTags(contextualParameters);
        
        for (Entry<String, Category> category : categories.entrySet())
        {
            CategoryCMSTag categoryTag = _transformToCMSTag(category.getValue(), contextualParameters);
            cmsTags.put(categoryTag.getId(), _transformToCMSTag(category.getValue(), contextualParameters));
        }
        
        return cmsTags;
    }
    
    public CMSTag getTag(String tagName, Map<String, Object> contextualParameters)
    {
        String categoryName = tagName;
        if (categoryName.startsWith(CategoryCMSTag.TAG_PREFIX))
        {
            categoryName = StringUtils.substringAfter(tagName, CategoryCMSTag.TAG_PREFIX);
            return _transformToCMSTag(_jcrCategoryProvider.getTag(categoryName, contextualParameters), contextualParameters);
        }
        return null;
    }
    
    public Collection<CMSTag> getTags(String tagName, Map<String, Object> contextualParameters)
    {
        String categoryName = tagName;
        if (categoryName.startsWith(CategoryCMSTag.TAG_PREFIX))
        {
            categoryName = StringUtils.substringAfter(tagName, CategoryCMSTag.TAG_PREFIX);
            
            Collection<Category> categories = _jcrCategoryProvider.getTags(categoryName, contextualParameters);
            
            return categories.stream()
                    .map(tag -> _transformToCMSTag(tag, contextualParameters))
                    .collect(Collectors.toList());
        }
        return List.of();
    }
    
    public boolean hasTag(String tagName, Map<String, Object> contextualParameters)
    {
        String categoryName = tagName;
        if (categoryName.startsWith(CategoryCMSTag.TAG_PREFIX))
        {
            categoryName = StringUtils.substringAfter(tagName, CategoryCMSTag.TAG_PREFIX);
            return _jcrCategoryProvider.hasTag(categoryName, contextualParameters);
        }
        return false;
    }
    
    /**
     * Transform a {@link Category} in a {@link CategoryCMSTag}, forcing visibility to public and target to CONTENT
     * @param category the category to transform
     * @param contextualParameters the contextual parameters
     * @return a {@link CategoryCMSTag} with the same values
     */
    protected CategoryCMSTag _transformToCMSTag(Category category, Map<String, Object> contextualParameters)
    {
        if (category != null)
        {
            CMSTag parentTag = null;
            Category parentCategory = category.getParent();
            if (parentCategory != null)
            {
                parentTag = this.getTag(parentCategory.getName(), contextualParameters);
            }
            
            String typeName = "CONTENT";
            TagTargetType targetType = _targetTypeEP.getTagTargetType(typeName);
            
            return new CategoryCMSTag(category, parentTag, targetType);
        }
        else
        {
            return null;
        }
    }
    
    @Override
    public List<String> getCSSUrls(Map<String, Object> contextualParameters)
    {
        List<String> cssUrls = super.getCSSUrls(contextualParameters);
        cssUrls.add("/plugins/workspaces/project-categories.min.css");
        
        return cssUrls;
    }
}
