/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.dav;

import java.util.Arrays;
import java.util.Collections;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.cocoon.ResourceNotFoundException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.authentication.AuthorizationRequiredException;

/**
 * Override of org.ametys.cms.rights.CheckReadAccessAction to throw a 404 not found if object is not found, instead of 500
 */
public class CheckReadAccessAction extends org.ametys.cms.rights.CheckReadAccessAction
{
    private static List<String> CHECK_PARENT_ACCESS_METHODS = Collections.unmodifiableList(Arrays.asList(new String[] {"PUT", "MKCOL"}));

    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        String method = request.getMethod();
        UserIdentity user = _currentUserProvider.getUser();
        
        // Get the Ametys object from either the parameters or the request attributes.
        AmetysObject ao = getAmetysObject(parameters, request);
        if (ao == null)
        {
            if (CHECK_PARENT_ACCESS_METHODS.contains(method))
            {
                String path = parameters.getParameter("path");
                Project project = (Project) request.getAttribute("project"); 
                ao = _getFirstParent(project, path);
            }
            else
            {
                throw new ResourceNotFoundException("Resource not found for path " + parameters.getParameter("objectPath", ""));
            }
        }
        
        if (_rightManager.hasAnonymousReadAccess(ao))
        {
            // Anonymous access allowed
            return EMPTY_MAP;
        }
        else if (user == null)
        {
            // User not yet authenticated
            throw new AuthorizationRequiredException();
        }
        else if (_rightManager.hasReadAccess(user, ao))
        {
            // User has read access
            return null;
        }
        
        // User is not authorized
        throw new AccessDeniedException("Access to object " + ao.getId() + " is not allowed for user " + user);
    }
    
    private AmetysObject _getFirstParent(Project project, String path)
    {
        ModifiableResourceCollection parent = _resolver.resolveByPath(project.getPath() + "/ametys-internal:resources/");
        for (String folder : path.split("/"))
        {
            if (folder.isEmpty())
            {
                continue;
            }
            if (!parent.hasChild(folder))
            {
                return parent;
            }
            parent = parent.getChild(folder);
        }
        return null;
    }
}
