/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.mentions.MentionUtils;
import org.ametys.core.observation.Event;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.explorer.rights.ResourceRightAssignmentContext;
import org.ametys.plugins.repository.ModifiableTraversableAmetysObject;
import org.ametys.plugins.workspaces.AbstractWorkspaceDAO;
import org.ametys.plugins.workspaces.comments.CommentsDAO;
import org.ametys.plugins.workspaces.documents.jcr.File;

/**
 * DAO for resources of a project
 */
public class WorkspaceExplorerResourceCommentsDAO extends AbstractWorkspaceDAO
{
    /** Right id to comment a resource */
    public static final String RIGHTS_RESOURCE_COMMENT = "Plugin_Explorer_File_Comment";

    /** The task JSON helper */
    protected DocumentsJSONHelper _documentsJSONHelper;
    
    /** The comments DAO */
    protected CommentsDAO _commentsDAO;
    
    /** The mentions helper */
    protected MentionUtils _mentionUtils;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _documentsJSONHelper = (DocumentsJSONHelper) manager.lookup(DocumentsJSONHelper.ROLE);
        _commentsDAO = (CommentsDAO) manager.lookup(org.ametys.cms.repository.comment.CommentsDAO.ROLE);
        _mentionUtils = (MentionUtils) manager.lookup(MentionUtils.ROLE);
    }

    
    /**
     * Comment a resource
     * @param resourceId the resource id
     * @param commentText the comment text
     * @return The resource data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> addComment(String resourceId, String commentText)
    {
        File resource = _resolver.resolveById(resourceId);
        
        // Check user right
        ModifiableTraversableAmetysObject resourcesRoot = resource.getParent();
        UserIdentity currentUser = _currentUserProvider.getUser();
        
        _checkUserRights(resourcesRoot, RIGHTS_RESOURCE_COMMENT);
        
        Comment comment = createComment(resource, commentText, resourcesRoot);
        
        
        // Notify listeners
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_ID, resourceId);
        
        eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_FILE_COMMENT, _mentionUtils.transformTextToReadableText(commentText, null));
        eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_FILE_COMMENT_ID, comment.getId());
        _observationManager.notify(new Event(org.ametys.plugins.workspaces.ObservationConstants.EVENT_RESOURCE_COMMENTED, currentUser, eventParams));


        List<Comment> comments = resource.getComments(true, true);
        return _documentsJSONHelper.commentsToJson(comments, resourceId, resourceId);
    }
    
    /**
     * Get comments
     * @param resourceId the resource id
     * @return The resource data
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ResourceRightAssignmentContext.ID)
    public List<Map<String, Object>> getComments(String resourceId)
    {
        File resource = _resolver.resolveById(resourceId);
        
        List<Comment> comments = resource.getComments(true, true);
        return _documentsJSONHelper.commentsToJson(comments, resourceId, resourceId);
    }
    
    /**
     * Edit a resource comment
     * @param resourceId the resource id
     * @param commentId the comment Id
     * @param commentText the comment text
     * @return The resource data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> editCommentResource(String resourceId, String commentId, String commentText)
    {
        File resource = _resolver.resolveById(resourceId);
        
        ModifiableTraversableAmetysObject resourcesRoot = resource.getParent();
        
        _checkUserRights(resourcesRoot, RIGHTS_RESOURCE_COMMENT);
        
        editComment(resource, commentId, commentText, resourcesRoot);

        List<Comment> comments = resource.getComments(true, true);
        return _documentsJSONHelper.commentsToJson(comments, resourceId, resourceId);
    }
    
    /**
     * Answer to a resource's comment
     * @param resourceId the resource id
     * @param commentId the comment id
     * @param commentText the comment text
     * @return The resource data
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public List<Map<String, Object>> answerCommentResource(String resourceId, String commentId, String commentText)
    {
        File resource = _resolver.resolveById(resourceId);
        
        ModifiableTraversableAmetysObject resourcesRoot = resource.getParent();

        _checkUserRights(resourcesRoot, RIGHTS_RESOURCE_COMMENT);
        
        answerComment(resource, commentId, commentText, resourcesRoot);

        List<Comment> comments = resource.getComments(true, true);
        return _documentsJSONHelper.commentsToJson(comments, resourceId, resourceId);
    }
}
