/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.indexing.solr;

import java.io.IOException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.UpdateResponse;

import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.QuerySyntaxException;
import org.ametys.cms.search.solr.SolrClientProvider;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.workspaces.search.query.ProjectQuery;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.WebConstants;

/**
 * Component responsible for indexing workspaces' objects
 */
public class SolrProjectIndexer extends AbstractLogEnabled implements Component, Serviceable, SolrWorkspacesConstants
{
    /** The avalon role. */
    public static final String ROLE = SolrProjectIndexer.class.getName();
    
    /** The Solr client provider */
    protected SolrClientProvider _solrClientProvider;
    /** The Solr indexer */
    protected SolrIndexer _solrIndexer;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _solrClientProvider = (SolrClientProvider) manager.lookup(SolrClientProvider.ROLE);
        _solrIndexer = (SolrIndexer) manager.lookup(SolrIndexer.ROLE);
    }
    
    /**
     * Unindex all Solr documents related to the given project in all workspaces and commit.
     * @param projectId The id of the project
     * @throws SolrServerException  if there is an error on the solr server
     * @throws IOException If there is a low-level I/O error.
     * @throws QuerySyntaxException If there is a low-level I/O error.
     */
    public void unindexProject(String projectId) throws SolrServerException, IOException, QuerySyntaxException
    {
        unindexProject(projectId, RepositoryConstants.DEFAULT_WORKSPACE);
        unindexProject(projectId, WebConstants.LIVE_WORKSPACE);
    }
    /**
     * Unindex all Solr documents related to the given project
     * @param projectId The id of the project
     * @param workspaceName The workspace name
     * @throws SolrServerException  if there is an error on the solr server
     * @throws IOException If there is a low-level I/O error.
     * @throws QuerySyntaxException If there is a low-level I/O error.
     */
    public void unindexProject(String projectId, String workspaceName) throws SolrServerException, IOException, QuerySyntaxException
    {
        Query query = new ProjectQuery(projectId);
        SolrClient solrClient = _solrClientProvider.getUpdateClient(workspaceName);
        UpdateResponse solrResponse = solrClient.deleteByQuery(_solrClientProvider.getCollectionName(workspaceName), query.build());
        int status = solrResponse.getStatus();
        
        if (status != 0)
        {
            throw new IOException("Ametys unindexing all Solr documents related to a Project- Expecting status code of '0' in the Solr response but got : '" + status + "'. Project id : " + projectId);
        }
        
        getLogger().debug("Successful unindexing of all Solr documents with field {} equals to {} in workspace {}", SolrWorkspacesConstants.PROJECT_ID, projectId, workspaceName);
    }
}
