/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.indexing.solr;

import java.io.IOException;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.solr.client.solrj.SolrClient;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;

import org.ametys.cms.content.indexing.solr.SolrIndexer;
import org.ametys.cms.content.indexing.solr.SolrResourceIndexer;
import org.ametys.cms.search.solr.SolrClientProvider;
import org.ametys.core.schedule.progression.ProgressionTrackerFactory;
import org.ametys.core.schedule.progression.SimpleProgressionTracker;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.web.indexing.solr.SolrWebFieldNames;
import org.ametys.web.repository.site.Site;

/**
 * Component responsible for indexing a project resource (document)
 */
public class SolrProjectResourceIndexer extends AbstractLogEnabled implements Component, Serviceable, SolrWorkspacesConstants
{
    /** The avalon role. */
    public static final String ROLE = SolrProjectResourceIndexer.class.getName();
    
    /** The Solr client provider */
    protected SolrClientProvider _solrClientProvider;
    /** Solr resource indexer */
    protected SolrResourceIndexer _solrResourceIndexer;
    /** Solr indexer */
    protected SolrIndexer _solrIndexer;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _solrClientProvider = (SolrClientProvider) manager.lookup(SolrClientProvider.ROLE);
        _solrResourceIndexer = (SolrResourceIndexer) manager.lookup(SolrResourceIndexer.ROLE);
        _solrIndexer = (SolrIndexer) manager.lookup(SolrIndexer.ROLE);
    }
    
    /**
     * Index the children project resources of the given resource collection
     * @param collection The collection of project resources
     * @param project The project whose resources are attached
     * @param workspaceName The workspace name
     * @throws Exception if an error occurs when processing the indexation of the project resources
     */
    public void indexProjectResources(ResourceCollection collection, Project project, String workspaceName) throws Exception
    {
        SolrClient solrClient = _solrClientProvider.getUpdateClient(workspaceName, true);
        indexProjectResources(collection, project, solrClient);
    }
    
    /**
     * Index the children project resources of the given resource collection
     * @param collection The collection of project resources
     * @param project The project whose resources are attached
     * @param solrClient The solr client to use
     * @throws Exception if an error occurs when processing the indexation of the project resources
     */
    public void indexProjectResources(ResourceCollection collection, Project project, SolrClient solrClient) throws Exception
    {
        indexProjectResources(collection, project, solrClient, ProgressionTrackerFactory.createSimpleProgressionTracker("Index project '" + project.getName() + "' resources", getLogger()));
    }
    
    /**
     * Index the children project resources of the given resource collection
     * @param collection The collection of project resources
     * @param project The project whose resources are attached
     * @param solrClient The solr client to use
     * @param progressionTracker The progression tracker
     * @throws Exception if an error occurs when processing the indexation of the project resources
     */
    public void indexProjectResources(ResourceCollection collection, Project project, SolrClient solrClient, SimpleProgressionTracker progressionTracker) throws Exception
    {
        if (collection == null)
        {
            return;
        }
        
        try (AmetysObjectIterable<AmetysObject> children = collection.getChildren())
        {
            long nbOfDocuments = children.getSize();
            
            progressionTracker.setSize(nbOfDocuments);
            for (AmetysObject object : children)
            {
                if (object instanceof ResourceCollection)
                {
                    indexProjectResources((ResourceCollection) object, project, solrClient);
                }
                else if (object instanceof Resource)
                {
                    _indexProjectResource((Resource) object, project, solrClient);
                }
                
                progressionTracker.increment();
            }
        }
    }
    
    /**
     * Index a project resource
     * @param resource The project resource (document)
     * @param project The project whose the resource is attached
     * @param workspaceName The workspace name
     * @throws Exception if an error occurs when processing the indexation of the project resource
     */
    public void indexProjectResource(Resource resource, Project project, String workspaceName) throws Exception
    {
        SolrClient solrClient = _solrClientProvider.getUpdateClient(workspaceName, true);
        _indexProjectResource(resource, project, solrClient);
    }
    
    private void _indexProjectResource(Resource resource, Project project, SolrClient solrClient) throws Exception
    {
        SolrInputDocument solrDocument = new SolrInputDocument();
        
        // Prepare resource doc
        _indexProjectResource(resource, solrDocument, project);
        
        // Indexation of the Solr document
        _indexResourceSolrDocument(resource, solrDocument, solrClient);
    }
    
    private void _indexProjectResource(Resource resource, SolrInputDocument document, Project project) throws Exception
    {
        
        _solrResourceIndexer.indexResource(resource, document, TYPE_PROJECT_RESOURCE);
        
        Site site = project.getSite();
        
        if (site != null)
        {
            // site name - Store.YES, Index.NOT_ANALYZED
            document.addField(SolrWebFieldNames.SITE_NAME, site.getName());
            
            // site type - Store.YES, Index.NOT_ANALYZED
            document.addField(SolrWebFieldNames.SITE_TYPE, site.getType());
        }
        
        document.addField(PROJECT_ID, project.getId());
        document.setField(KEYWORDS, resource.getKeywords());
    }
    
    private void _indexResourceSolrDocument(Resource resource, SolrInputDocument document, SolrClient solrClient) throws SolrServerException, IOException
    {
        UpdateResponse solrResponse = solrClient.add(_solrClientProvider.getCollectionName(), document);
        int status = solrResponse.getStatus();
        
        if (status != 0)
        {
            throw new IOException("Ametys resource indexing - Expecting status code of '0' in the Solr response but got : '" + status + "'. Resource id : " + resource.getId());
        }
        
        getLogger().debug("Successful resource indexing. Resource identifier : {}", resource.getId());
    }
}
