/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.initialization;

import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.ModifiableSitemapElement;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Page initializer that lookup for the projects catalogue service.
 * The class tries to configure the service using the site configuration
 */
public class ProjectsCatalogPageInitializer extends DefaultStaticPageInitializer
{
    @Override
    public Optional< ? extends Page> createPage(ModifiableSitemapElement parent)
    {
        Optional< ? extends Page> result = super.createPage(parent);
        if (result.isPresent() && result.get() instanceof ModifiableSitemapElement page)
        {
            try (AmetysObjectIterable<ModifiableZone> zones = page.getZones())
            {
                for (ModifiableZone zone : zones)
                {
                    try (AmetysObjectIterable<ModifiableZoneItem> zoneItems = zone.getZoneItems())
                    {
                        for (ModifiableZoneItem zoneItem : zoneItems)
                        {
                            if (StringUtils.equals(zoneItem.getServiceId(), "org.ametys.plugins.workspaces.service.ProjectsCatalogue"))
                            {
                                ModifiableModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
                                try (AmetysObjectIterable<Sitemap> sitemaps = page.getSite().getSitemaps())
                                {
                                    serviceParameters.setValue("availableLanguages", sitemaps.stream().map(Sitemap::getName).toArray(String[]::new));
                                }
                                zoneItem.saveChanges();
                                
                                // FIXME we should use api that manage the event instead of doing it manually
                                Map<String, Object> eventParams = new HashMap<>();
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM, zoneItem);
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM_ID, zoneItem.getId());
                                eventParams.put(ObservationConstants.ARGS_ZONE_ITEM_SERVICE, zoneItem.getServiceId());
                                _observationManager.notify(new Event(ObservationConstants.EVENT_SERVICE_MODIFIED, _currentUserProvider.getUser(), eventParams));
                            }
                        }
                    }
                }
            }
            catch (Exception e)
            {
                getLogger().error("An unexpected error occurred during the page initialization.", e);
            }
        }
        return result;
    }
}
