/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.minisite;

import java.util.Collection;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.apache.commons.lang3.ArrayUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.repository.page.Page;

/**
 * This {@link Observer} throws a minisite page event updated when a content has been validated
 */
public class ConvertContentValidatedEvent2MinisitePageEventObserver extends AbstractConvertMinisiteEventObserver
{
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_CONTENT_VALIDATED);
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
        String[] cTypes = content.getTypes();
        if (ArrayUtils.contains(cTypes, WorkspacesConstants.PROJECT_ARTICLE_CONTENT_TYPE) && content instanceof WebContent)
        {
            Project project = _getProject((WebContent) content);
            if (project != null)
            {
                _notifyPageUpdate(content, project, event);
            }
        }
    }

    private void _notifyPageUpdate(Content content, Project project, Event event)
    {
        Collection<Page> referencingPages = ((WebContent) content).getReferencingPages();
        for (Page page : referencingPages)
        {
            Map<String, Object> eventParams = new HashMap<>();
            eventParams.put(org.ametys.web.ObservationConstants.ARGS_PAGE, page);
            eventParams.put(org.ametys.plugins.workspaces.ObservationConstants.ARGS_PROJECT, project);
            
            _observationManager.notify(new Event(org.ametys.plugins.workspaces.ObservationConstants.EVENT_MINISITE_PAGE_UPDATED, event.getIssuer(), eventParams));
        }
    }

    private Project _getProject(WebContent content)
    {
        List<String> projects = _projectManager.getProjectsForSite(content.getSiteName());
        if (projects.size() > 0)
        {
            return _projectManager.getProject(projects.get(0));
        }
        return null;
    }
}
