/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.observers;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.configuration.ConfigurationException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.xml.sax.SAXException;

import org.ametys.core.observation.Event;
import org.ametys.core.observation.ObservationManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.PagePopulator;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.ObservationConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.sitemap.Sitemap;

/**
 * Initializes all project workspace pages when a sitemap is created.
 */
public class InitializeProjectSitemapObserver extends AbstractInitializeProjectObserver
{
    /** The observation manager. */
    protected ObservationManager _observationManager;
    
    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** the page populator */
    protected PagePopulator _pagePopulator;
    /** the plugin name */
    protected String _pluginName;

    @Override
    public void setPluginInfo(String pluginName, String featureName, String id)
    {
        _pluginName = pluginName;
    }
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _observationManager = (ObservationManager) manager.lookup(ObservationManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _pagePopulator = (PagePopulator) manager.lookup(PagePopulator.ROLE);
    }

    @Override
    public void doObserve(Event event, Map<String, Object> transientVars, Site site, Project project) throws Exception
    {
        if (project != null)
        {
            // Initialize each sitemap with a set of predefined and configured pages
            try (AmetysObjectIterable<Sitemap> sitemaps = site.getSitemaps())
            {
                for (Sitemap sitemap : sitemaps)
                {
                    try (AmetysObjectIterable< ? extends Page> pages = sitemap.getChildrenPages())
                    {
                        if (pages.getSize() == 0)
                        {
                            _initializeSitemap(sitemap, project);
                        }
                    }
                }
            }
        }
    }
    
    
    /**
     * Initialize the given sitemap.
     * @param sitemap the Sitemap object.
     * @param project the corresponding project
     */
    protected void _initializeSitemap(Sitemap sitemap, Project project)
    {
        String path = "skin:" + sitemap.getSite().getSkinId() + "://conf/project-home-model.xml";
        try
        {
            _pagePopulator.initPage(sitemap, path);
        }
        catch (ConfigurationException | IOException | SAXException e)
        {
            getLogger().error("An error occurred while trying to create and configure the project home page.", e);
        }
        
        _projectManager.initializeModulesSitemap(project, sitemap);
        
        // Notify of the sitemap change.
        Map<String, Object> eventParams = new HashMap<>();
        eventParams.put(ObservationConstants.ARGS_SITEMAP, sitemap);
        _observationManager.notify(new Event(ObservationConstants.EVENT_SITEMAP_UPDATED, _currentUserProvider.getUser(), eventParams));
    }
}
