/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryHelper;
import org.ametys.plugins.workspaces.categories.CategoryProviderExtensionPoint;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.WebConstants;
import org.ametys.web.WebHelper;
import org.ametys.web.repository.site.SiteManager;

/**
 * Abstract generator for search modules
 *
 */
public abstract class AbstractSearchModuleGenerator extends ServiceableGenerator
{
    /** The ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The project member manager */
    protected ProjectMemberManager _projectMembers;
    /** CategoryProviderExtensionPoint */
    protected CategoryProviderExtensionPoint _categoryProviderEP;
    /** The category helper */
    protected CategoryHelper _categoryHelper;
    /** The user helper */
    protected UserHelper _userHelper;
    /** The site manager */
    protected SiteManager _siteManager;
    /** WorkspaceModuleExtensionPoint */
    protected WorkspaceModuleExtensionPoint _workspaceModuleEP;
    /** Right manager */
    protected RightManager _rightManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
        _projectMembers = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
        _categoryProviderEP = (CategoryProviderExtensionPoint) smanager.lookup(CategoryProviderExtensionPoint.ROLE);
        _categoryHelper = (CategoryHelper) smanager.lookup(CategoryHelper.ROLE);
        _userHelper = (UserHelper) smanager.lookup(UserHelper.ROLE);
        _siteManager = (SiteManager) smanager.lookup(SiteManager.ROLE);
        _workspaceModuleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        String siteName = WebHelper.getSiteName(request);
        String lang = request.getParameter("lang");
        String textfield = request.getParameter("textfield");
        
        request.setAttribute(WebConstants.REQUEST_ATTR_SITEMAP_NAME, lang);
        request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, _siteManager.getSite(siteName).getSkinId());
        
        int offset = parameters.getParameterAsInteger("offset", 0);
        int limit = parameters.getParameterAsInteger("limit", 10);
        int minLimit = parameters.getParameterAsInteger("minLimit", 10);
        
        contentHandler.startDocument();
        
        saxHits(siteName, lang, textfield, request, offset, limit, minLimit);
        
        contentHandler.endDocument();
    }
    
    /**
     * Sax the results after search
     * @param results list of results
     * @param lang the current language
     * @param offset the start of search
     * @param limit the max number of results
     * @param minLimit the min number of results
     * @param totalCount total count available
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxHits(List<? extends AmetysObject> results, String lang, int offset, int limit, int minLimit, long totalCount) throws SAXException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("count", String.valueOf(results.size()));
        attrs.addCDATAAttribute("total", String.valueOf(totalCount));
        attrs.addCDATAAttribute("offset", String.valueOf(offset));
        attrs.addCDATAAttribute("limit", String.valueOf(limit));
        attrs.addCDATAAttribute("minLimit", String.valueOf(minLimit));
        
        XMLUtils.startElement(contentHandler, "hits", attrs);
        
        for (AmetysObject result : results)
        {
            try
            {
                saxHit(result, lang);
            }
            catch (Exception e)
            {
                getLogger().error("Unable to sax result for object of id '" + result.getId() + "'", e);
            }
        }
        
        XMLUtils.endElement(contentHandler, "hits");
    }
    
    /**
     * Get the project holding the object.
     * Be careful, use only for objects that belongs to a project (thread, task, event, ...)
     * @param ao the Ametys object
     * @return the project or null if nor found
     */
    protected Project getProject(AmetysObject ao)
    {
        AmetysObject parent = ao.getParent();
        
        while (parent != null && !(parent instanceof Project))
        {
            parent = parent.getParent();
        }
        
        return parent != null ? (Project) parent : null;
    }
    
    /**
     * SAX a project
     * @param project the project
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxProject(Project project) throws SAXException
    {
        if (project != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", project.getName());
            XMLUtils.startElement(contentHandler, "project", attrs);
            XMLUtils.createElement(contentHandler, "title", project.getTitle());
            Category category = project.getCategories()
                .stream()
                .findFirst()
                .map(c -> _categoryProviderEP.getTag(c, null))
                .orElse(null);
            saxCategory(category);
            XMLUtils.endElement(contentHandler, "project");
            
        }
    }
    
    /**
     * SAX category
     * @param category the category
     * @throws SAXException if an error occured while saxing
     */
    protected void saxCategory(Category category) throws SAXException
    {
        if (category != null)
        {
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("name", category.getName());
            XMLUtils.startElement(contentHandler, "category", attrs);
            category.getTitle().toSAX(contentHandler, "title");
            saxCategoryColor(category);
            XMLUtils.endElement(contentHandler, "category");
        }
    }
    
    /**
     * SAX the category color
     * @param category the category
     * @throws SAXException if an error occured while saxing
     */
    protected void saxCategoryColor(Category category) throws SAXException
    {
        Map<String, String> colors = _categoryHelper.getCategoryColor(category);
        
        XMLUtils.startElement(contentHandler, "color");
        colors.entrySet().stream()
            .forEach(LambdaUtils.wrapConsumer(entry -> XMLUtils.createElement(contentHandler, entry.getKey(), entry.getValue())));
        XMLUtils.endElement(contentHandler, "color");
    }
    
    /**
     * Get the project names targeted by the search
     * @param request the request
     * @param userOnly true to get user's projects only
     * @return the project names
     */
    public List<Project> getProjects(Request request, boolean userOnly)
    {
        List<String> projectNames = Arrays.asList(request.getParameter("project").split(","))
                .stream()
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toList());
        
        Set<String> filteredCategories = getCategories(request);
        
        if (projectNames.size() > 0)
        {
            return projectNames
                .stream()
                .map(p -> _projectManager.getProject(p))
                .filter(p -> !Collections.disjoint(p.getCategories(), filteredCategories))
                .collect(Collectors.toList());
        }
        else if (userOnly)
        {
            // Get user projects
            UserIdentity user = _currentUserProvider.getUser();
            
            return  _projectManager.getUserProjects(user, filteredCategories)
                    .keySet()
                    .stream()
                    .collect(Collectors.toList());
        }
        else
        {
            // All projects
            return  _projectManager.getProjects(filteredCategories);
        }
    }
    
    /**
     * Filter a list of projects to return only those where the module is available
     * @param projects list of projects
     * @param moduleId id of the module to search for read access
     * @return a filtered list of projects
     */
    protected List<Project> filterProjectsForModule(List<Project> projects, String moduleId)
    {
        WorkspaceModule module = _workspaceModuleEP.getModule(moduleId);
        
        if (module == null || projects == null || projects.isEmpty())
        {
            return Collections.EMPTY_LIST;
        }
        
        List<Project> filteredProjects = projects.stream()
                .filter(project -> _projectManager.isModuleActivated(project, module.getId()))
                .filter(project -> _rightManager.currentUserHasReadAccess(module.getModuleRoot(project, false)))
                .collect(Collectors.toList());
        
        return filteredProjects;
    }
    
    /**
     * Get the category names targeted by the search
     * @param request the request
     * @return the category names
     */
    protected Set<String> getCategories(Request request)
    {
        List<String> categorieNames = Arrays.asList(request.getParameter("category").split(","))
                .stream()
                .filter(StringUtils::isNotEmpty)
                .collect(Collectors.toList());
        
        if (categorieNames.isEmpty())
        {
            // Get all leaf categories
            return _categoryHelper.getLeafCategories()
                .stream()
                .map(Category::getName)
                .collect(Collectors.toSet());
        }
        else
        {
            Set<String> filteredCategories = new HashSet<>();
            for (String categoryName : categorieNames)
            {
                Category category = _categoryProviderEP.getTag(categoryName, Collections.EMPTY_MAP);
                if (category != null)
                {
                    filteredCategories.addAll(_categoryHelper.getLeafCategories(category)
                            .stream()
                            .map(Category::getName)
                            .collect(Collectors.toSet()));
                }
            }
            
            return filteredCategories;
        }
    }
    
    /**
     * Sax the results
     * @param siteName the current site name
     * @param lang the current language
     * @param textfield the search inputs
     * @param request the request
     * @param offset the start of search
     * @param limit the max number of results
     * @param minLimit the min number of results
     * @throws SAXException if an error occurred while saxing
     * @throws ProcessingException if the search failed
     */
    protected abstract void saxHits(String siteName, String lang, String textfield, Request request, int offset, int limit, int minLimit) throws SAXException, ProcessingException;

    /**
     * Sax the content hit
     * @param object the AmetysObject
     * @param lang the language
     * @throws Exception if an error occurred while saxing result
     */
    protected abstract void saxHit(AmetysObject object, String lang) throws Exception;
    
    /**
     * SAX a user identity
     * @param userIdentity the user identity
     * @param tagName the tag name
     * @throws SAXException if an error occurred while saxing
     */
    protected void saxUser(UserIdentity userIdentity, String tagName) throws SAXException
    {
        if (userIdentity != null)
        {
            _userHelper.saxUserIdentity(userIdentity, contentHandler, tagName);
        }
    }
    
}
