/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.ContentLanguageQuery;
import org.ametys.cms.search.query.ContentTypeQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.query.StringQuery;
import org.ametys.cms.search.query.TagQuery;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.categories.Category;
import org.ametys.plugins.workspaces.categories.CategoryCMSTag;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.repository.content.WebContent;
import org.ametys.web.search.query.SiteQuery;

/**
 * Generator for posts search module
 *
 */
public class NewsSearchModuleGenerator extends AbstractContentSolrSearchModuleGenerator
{
    private ContentTypesHelper _ctypeHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _ctypeHelper = (ContentTypesHelper) smanager.lookup(ContentTypesHelper.ROLE);
    }
    
    @Override
    protected SimpleContentSearcher getSearcher()
    {
        return _contentSearcherFactory.create(WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID, WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID);
    }
    
    @Override
    protected String getSortFieldName()
    {
        return "lastModified";
    }
    
    @Override
    protected SortOrder getSortOrder()
    {
        return SortOrder.DESC;
    }
    
    @Override
    protected Query getQuery(String siteName, String lang, String textfield, Request request)
    {
        Collection<Query> queries = new HashSet<>();
        
        if (StringUtils.isNotBlank(textfield))
        {
            Query titleQuery = new StringQuery(SolrFieldNames.TITLE, Operator.SEARCH, textfield.trim(), lang);
            Query fullTextQuery = new FullTextQuery(textfield.trim(), SolrFieldNames.FULL, lang, Operator.SEARCH);
            queries.add(new OrQuery(titleQuery, fullTextQuery));
        }
        
        List<Project> projects = getProjects(request, true);
        Query projectNewsQuery = getProjectNewsQuery(request, lang, projects);
        
        boolean noProjectSelected = StringUtils.isBlank(request.getParameter("project"));
        if (noProjectSelected)
        {
            Query catalogNewsQuery = getCatalogNewsQuery(request, siteName);
            queries.add(new OrQuery(projectNewsQuery, catalogNewsQuery));
        }
        else
        {
            queries.add(projectNewsQuery);
        }
        
        return new AndQuery(queries);
    }
    
    /**
     * Get the query for projects' news
     * @param request the request
     * @param lang the language
     * @param projects the targeted projects
     * @return the query
     */
    protected Query getProjectNewsQuery(Request request, String lang, List<Project> projects)
    {
        Query projectNewsTypeQuery = new ContentTypeQuery(Operator.EQ,  WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID);
        Query projectSiteQuery = getSiteQuery(projects);
        Query langQuery = new ContentLanguageQuery(lang);
        
        return new AndQuery(projectNewsTypeQuery, projectSiteQuery, langQuery);
    }
    
    /**
     * Get the query for catalog's news
     * @param request the request
     * @param siteName the current site name
     * @return the query
     */
    protected Query getCatalogNewsQuery(Request request, String siteName)
    {
        Collection<Query> queries = new HashSet<>();
        
        queries.add(new ContentTypeQuery(Operator.EQ, WorkspacesConstants.CATALOG_NEWS_CONTENT_TYPE_ID));
        queries.add(new SiteQuery(siteName));
        
        Set<String> categories = getCategories(request);
        if (categories != null && !categories.isEmpty())
        {
            List<String> tagCategories = categories.stream()
                .map(c -> CategoryCMSTag.TAG_PREFIX + c)
                .collect(Collectors.toList());
            
            queries.add(new TagQuery(tagCategories.toArray(new String[tagCategories.size()])));
        }
        
        return new AndQuery(queries);
    }
    
    @Override
    protected void saxAdditionalInformation(Content content) throws SAXException
    {
        super.saxAdditionalInformation(content);
        
        if (_ctypeHelper.isInstanceOf(content,  WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID))
        {
            // SAX project
            String siteName = ((WebContent) content).getSiteName();
            Project project = _projectManager.getProject(siteName);
            if (project != null)
            {
                saxProject(project);
            }
        }
        else
        {
            // SAX category
            saxCategory(content);
        }
    }
    
    /**
     * SAX category
     * @param content the news content
     * @throws SAXException if an error occured while saxing
     */
    protected void saxCategory(Content content) throws SAXException
    {
        Set<String> tags = content.getTags();
        
        for (String tagName : tags)
        {
            Category category = _categoryProviderEP.getTag(tagName, null);
            saxCategory(category);
        }
    }
    
}
