/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.time.LocalDate;
import java.util.List;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.FullTextExpression;
import org.ametys.plugins.repository.query.expression.OrExpression;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.tasks.Task;
import org.ametys.plugins.workspaces.tasks.TasksWorkspaceModule;
import org.ametys.plugins.workspaces.tasks.jcr.JCRTask;

/**
 * Generator for tasks search module
 */
public class TaskSearchModuleGenerator extends AbstractXpathSearchModuleGenerator
{
    /** Task Module */
    protected TasksWorkspaceModule _taskModule;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _taskModule = moduleManagerEP.getModule(TasksWorkspaceModule.TASK_MODULE_ID);
    }
    
    @Override
    protected String getXPathQuery(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception
    {
        List<Project> projects = getProjects(request, true);
        
        List<Project> filteredProjects = filterProjectsForModule(projects, TasksWorkspaceModule.TASK_MODULE_ID);
        if (filteredProjects.isEmpty())
        {
            // No project available for this module
            return null;
        }
        
        String projectXPathQuery = getProjectXPathQuery(filteredProjects);
        
        String searchQuery = "";
        if (StringUtils.isNotBlank(textfield))
        {
            Expression labelExpr = new FullTextExpression("label", textfield);
            Expression descExpr = new FullTextExpression("description", textfield);
            
            OrExpression expr = new OrExpression(labelExpr, descExpr);
            searchQuery = "[" + expr.build() + "]";
        }
        
        String jcrQuery = projectXPathQuery + "//element(*, ametys:task)" + searchQuery;
        
        return jcrQuery;
    }
    
    @Override
    protected void saxHit(AmetysObject object, String lang) throws Exception
    {
        if (object instanceof Task)
        {
            Task task = (Task) object;
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute(JCRTask.ATTRIBUTE_TASK_ID, task.getId());
            XMLUtils.startElement(contentHandler, "hit", attrs);
            
            XMLUtils.createElement(contentHandler, JCRTask.ATTRIBUTE_LABEL, task.getLabel());
            XMLUtils.createElement(contentHandler, JCRTask.ATTRIBUTE_DESCRIPTION, task.getDescription());
            
            //uri
            Project project = getProject(object);
            String taskUri = _taskModule.getTaskUri(project, task.getId());
            XMLUtils.createElement(contentHandler, "uri", taskUri);
            
            XMLUtils.createElement(contentHandler, JCRTask.ATTRIBUTE_ISCLOSED, String.valueOf(task.isClosed()));
            
            LocalDate endDate = task.getDueDate();
            if (endDate != null)
            {
                XMLUtils.createElement(contentHandler, JCRTask.ATTRIBUTE_DUEDATE, DateUtils.localDateToString(endDate));
            }
            
            UserIdentity author = task.getAuthor();
            saxUser(author, JCRTask.ATTRIBUTE_AUTHOR);
            
            XMLUtils.startElement(contentHandler, JCRTask.ATTRIBUTE_ASSIGNMENTS);
            List<UserIdentity> assignments = task.getAssignments();
            for (UserIdentity assignment : assignments)
            {
                saxUser(assignment, "assignment");
            }
            XMLUtils.endElement(contentHandler, JCRTask.ATTRIBUTE_ASSIGNMENTS);
            
            saxProject(project);
            
            XMLUtils.endElement(contentHandler, "hit");
            
        }
    }

}
