/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.query;

import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.cocoon.environment.Request;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;
import org.ametys.plugins.workspaces.project.ProjectWorkspaceSiteType;
import org.ametys.web.frontoffice.AbstractSearchGenerator;
import org.ametys.web.frontoffice.FrontOfficeSearcherFactory.QueryFacet;
import org.ametys.web.frontoffice.QueryAdapterFOSearch;
import org.ametys.web.indexing.solr.SolrWebFieldNames;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.search.query.SiteTypeQuery;

/**
 * The query adapter to handle the "with workspaces" FO search service parameter
 */
public class WorkspacesQueryAdapterFOSearch implements QueryAdapterFOSearch
{
    @Override
    public int getPriority()
    {
        return QueryAdapterFOSearch.MIN_PRIORITY;
    }
    
    @Override
    public Query modifyQuery(Query query, Request request, Collection<String> siteNames, String language)
    {
        return query;
    }

    @Override
    public Query modifySiteQueryFilter(Query siteQuery, Request request, Collection<String> siteNames, String language)
    {
        if (_withWorkspaces(request))
        {
            SiteTypeQuery siteTypeQuery = new SiteTypeQuery(ProjectWorkspaceSiteType.TYPE_ID);
            return new OrQuery(siteQuery, siteTypeQuery);
        }
        
        return siteQuery;
    }

    @Override
    public Query modifySitemapQueryFilter(Query sitemapQuery, Request request, Collection<String> siteNames, String language)
    {
        if (_withWorkspaces(request))
        {
            SiteTypeQuery siteTypeQuery = new SiteTypeQuery(ProjectWorkspaceSiteType.TYPE_ID);
            return new OrQuery(sitemapQuery, siteTypeQuery);
        }
        
        return sitemapQuery;
    }
    
    public Set<QueryFacet> modifyQueryFacets(Set<QueryFacet> queryFacets, Request request)
    {
        if (_withWorkspaces(request))
        {
            Set<QueryFacet> modifiedQueryFacets = new HashSet<>();
            for (QueryFacet queryFacet : queryFacets)
            {
                if (queryFacet.getKeyName().equals(AbstractSearchGenerator.DOCUMENT_TYPE_IS_PAGE_RESOURCE_FACET_NAME))
                {
                    // add "projectResource" to facet for resources
                    modifiedQueryFacets.add(new QueryFacet(AbstractSearchGenerator.DOCUMENT_TYPE_IS_PAGE_RESOURCE_FACET_NAME, 
                            SolrWebFieldNames.PAGE_CONTENT_TYPES, 
                            SolrFieldNames.DOCUMENT_TYPE + ":" + SolrWebFieldNames.TYPE_PAGE_RESOURCE + " OR " + SolrFieldNames.DOCUMENT_TYPE + ":" + SolrWorkspacesConstants.TYPE_PROJECT_RESOURCE));
                }
                else
                {
                    // no modification
                    modifiedQueryFacets.add(queryFacet);
                }
            }
            return modifiedQueryFacets;
        }
        
        return queryFacets;
    }
    
    @Override
    public void addDocumentType(List<String> documentTypes)
    {
        documentTypes.add(SolrWorkspacesConstants.TYPE_PROJECT_RESOURCE);
    }
    
    private boolean _withWorkspaces(Request request)
    {
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(ZoneItem.class.getName());
        if (zoneItem != null)
        {
            ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
            return serviceParameters.getValue("withWorkspaces", false, false);
        }
        
        return false;
    }

}
