/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.suggestions;

import java.io.IOException;
import java.util.Arrays;
import java.util.Collection;
import java.util.Comparator;
import java.util.List;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections.CollectionUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.helper.ProjectXsltHelper;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.project.objects.Project.InscriptionStatus;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.ZoneItem;

/**
 * Generator for suggested projects. The suggestion projects are:&lt;br/&gt;
 * - those of same categories OR with common keywords with the current project&lt;br/&gt;
 * - those the current user is not already a member&lt;br/&gt;
 * - sorted by common keywords
 */
public class ProjectsSuggestionsGenerator extends ServiceableGenerator
{
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The project memebers manager */
    protected ProjectMemberManager _projectMembers;
    /** The project catalog manager */
    protected ProjectsCatalogueManager _projectCatalogManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _projectManager = (ProjectManager) smanager.lookup(ProjectManager.ROLE);
        _projectMembers = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
        _projectCatalogManager = (ProjectsCatalogueManager) smanager.lookup(ProjectsCatalogueManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) smanager.lookup(CurrentUserProvider.ROLE);
    }
    
    public void generate() throws IOException, SAXException, ProcessingException
    {
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "suggestions");
        
        String projectName = ProjectXsltHelper.project();
        
        if (StringUtils.isEmpty(projectName))
        {
            getLogger().warn("There is no current project to get the project suggestions");
        }
        else
        {
            Request request = ObjectModelHelper.getRequest(objectModel);
            
            ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
            Long nbMax = 0L;
            
            if (zoneItem != null)
            {
                ModelAwareDataHolder serviceParameters = zoneItem.getServiceParameters();
                if (serviceParameters != null && serviceParameters.hasValue("maxlength"))
                {
                    nbMax = zoneItem.getServiceParameters().getValue("maxlength");
                }
            }
            
            List<Project> suggestedProjects = getSuggestedProjects(projectName, _currentUserProvider.getUser(), nbMax.equals(0L) ? Integer.MAX_VALUE : nbMax.intValue());
            
            
            for (Project suggestedProject : suggestedProjects)
            {
                _projectCatalogManager.saxProject(contentHandler, suggestedProject);
            }
        }
        
        XMLUtils.endElement(contentHandler, "suggestions");
        contentHandler.endDocument();
    }
    
    /**
     * Get the suggested projects for a given project and a user 
     * @param projectName the project name
     * @param user the user
     * @param max the max number of suggestions
     * @return the related projects
     */
    protected List<Project> getSuggestedProjects(String projectName, UserIdentity user, int max)
    {
        Project project = _projectManager.getProject(projectName);
        if (project != null)
        {
            Set<String> projectCategories = project.getCategories();
            Set<String> projectKeywords = Set.of(project.getKeywords());
            
            // Get the projects with common categories or common keywords
            List<Project> relatedProjects = _projectManager.getProjects(projectCategories, projectKeywords, true);
            
            List<Project> projects = relatedProjects.stream()
                .filter(p -> !p.getName().equals(projectName)) // exclude current project
                .filter(p -> p.getInscriptionStatus() != InscriptionStatus.PRIVATE) // public or moderate projects only
                .filter(p -> _projectMembers.getProjectMember(p, user) == null) // exclude user's project
                .filter(p -> _projectManager.isUserInProjectPopulations(p, user)) // exclude project without current user population
                .collect(Collectors.toList());
            
            // Sort by common categories then keywords with current project
            projects.sort(new ProjectComparator(projectCategories, projectKeywords));
            
            return projects.subList(0, Integer.min(projects.size(), max));
        }
        else
        {
            getLogger().warn("Unknown project with name '" + projectName + "'");
            return List.of();
        }
    }
    
    class ProjectComparator implements Comparator<Project>
    {
        private Collection<String> _baseKeywords;
        private Collection<String> _baseCategories;
        
        public ProjectComparator(Collection<String> baseCategories, Collection<String> baseKeywords)
        {
            _baseCategories = baseCategories;
            _baseKeywords = baseKeywords;
        }
        
        public int compare(Project p1, Project p2)
        {
            int score1 = countCommonCategories(p1) * 2 + countCommonKeywords(p1);
            int score2 = countCommonCategories(p2) * 2 + countCommonKeywords(p2);
            return score2 - score1;
        }
        
        int countCommonCategories(Project project)
        {
            Set<String> projectCategories = project.getCategories();
            
            if (_baseCategories.isEmpty() || projectCategories.isEmpty())
            {
                // no common categories
                return 0;
            }
            return CollectionUtils.intersection(_baseCategories, projectCategories).size();
        }
        
        int countCommonKeywords(Project project)
        {
            List<String> projectKeywords = Arrays.asList(project.getKeywords());
            
            if (_baseKeywords.isEmpty() || projectKeywords.isEmpty())
            {
                // no common keywords
                return 0;
            }
            return CollectionUtils.intersection(_baseKeywords, projectKeywords).size();
        }
        
    }

}
