/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.tasks;

import java.time.LocalDate;
import java.time.ZonedDateTime;
import java.util.List;

import org.ametys.cms.data.ametysobject.ModifiableModelAwareDataAwareAmetysObject;
import org.ametys.cms.repository.AttachableAmetysObject;
import org.ametys.cms.repository.CommentableAmetysObject;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.RemovableAmetysObject;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;

/**
 * Task interface
 */
public interface Task extends ModifiableModelAwareDataAwareAmetysObject, RemovableAmetysObject, TaggableAmetysObject, CommentableAmetysObject<Comment>, AttachableAmetysObject
{
    /**
     * The identifier of a task list
     * @return The id
     */
    public String getTaskListId();
    
    /**
     * Set the identifier of a tasks list
     * @param tasksListId The tasks list id
     */
    public void setTasksListId(String tasksListId);
    
    /**
     * The position into the task list
     * @return The position
     */
    public Long getPosition();
    
    /**
     * Set the position into the task list
     * @param position The position into the task list
     */
    public void setPosition(Long position);
    
    /**
     * The title of the task
     * @return The title
     */
    public String getLabel();
    
    /**
     * Set the label of the task
     * @param label The label
     */
    public void setLabel(String label);
    
    /**
     * The description of the task
     * @return The description
     */
    public String getDescription();
    
    /**
     * Set the description of the task
     * @param description The description
     */
    public void setDescription(String description);
    
    /**
     * The starting date of the task
     * @return The start date
     */
    public LocalDate getStartDate();
    
    /**
     * Set the starting date of the task
     * @param startDate The start date
     */
    public void setStartDate(LocalDate startDate);
    
    /**
     * The due date of the task
     * @return The due date
     */
    public LocalDate getDueDate();
    
    /**
     * Set the due date of the task
     * @param dueDate The due date
     */
    public void setDueDate(LocalDate dueDate);
    
    /**
     * True if the task is closed
     * @return <code>true</code> if the task is closed
     */
    public boolean isClosed();
    
    /**
     * Close the task
     * @param isClosed <code>true</code> to close the task
     */
    public void close(boolean isClosed);
    
    /**
     * Get the author who close the task
     * @return the author who close the task
     */
    public UserIdentity getCloseAuthor();

    /**
     * Set the author who close the task.
     * @param author the author
     */
    public void setCloseAuthor(UserIdentity author);
    
    /**
     * Get the task's date of closure
     * @return the task's date of closure
     */
    public LocalDate getCloseDate();
    
    /**
     * Set the task's date of closure
     * @param closedDate The date of closure
     */
    public void setCloseDate(LocalDate closedDate);
    
    /**
     * Get the author of the task
     * @return the author
     */
    public UserIdentity getAuthor();
    
    /**
     * Set the author of this task.
     * @param author the author
     */
    public void setAuthor(UserIdentity author);
    
    /**
     * Get the list of user assigned to this task
     * @return The assignment list
     */
    public List<UserIdentity> getAssignments();
    
    /**
     * Set the list of user assigned to this task
     * @param assignments The assignment list
     */
    public void setAssignments(List<UserIdentity> assignments);
    
    /**
     * Get the task's creation date.
     * @return the task's creation date.
     */
    public ZonedDateTime getCreationDate();
    
    /**
     * Set the post's creation date.
     * @param startDate the post's creation date.
     */
    public void setCreationDate(ZonedDateTime startDate);
    
    /**
     * Get the task's last modification date.
     * @return the task's last modification date.
     */
    public ZonedDateTime getLastModified();
    
    /**
     * Set the post's modification date.
     * @param date the last modification date to set.
     */
    public void setLastModified(ZonedDateTime date);
    
    /**
     * Get the task's check list
     * @return the task's check list
     */
    public List<CheckItem> getCheckList();
    
    /**
     * Set the list of check item
     * @param checkItems the list of check item
     */
    public void setCheckListItem(List<CheckItem> checkItems);
    
    /**
     * Object to represent a check item
     */
    public class CheckItem
    {
        private String _label;
        private boolean _isChecked;
        
        /**
         * The constructor
         * @param label the label
         * @param isChecked true is the item is check
         */
        public CheckItem(String label, boolean isChecked)
        {
            _label = label;
            _isChecked = isChecked;
        }
        
        /**
         * Get the label
         * @return the label
         */
        public String getLabel()
        {
            return _label;
        }
        
        /**
         * <code>true</code> if the item is checked
         * @return <code>true</code> if the item is checked
         */
        public boolean isChecked()
        {
            return _isChecked;
        }
    }
}
