/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.util;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.xml.sax.SAXParser;
import org.xml.sax.InputSource;

import org.ametys.cms.content.RichTextHandler;
import org.ametys.cms.data.Binary;
import org.ametys.cms.data.RichText;
import org.ametys.cms.repository.Content;
import org.ametys.cms.repository.ReactionableObject.ReactionType;
import org.ametys.cms.repository.comment.AbstractComment;
import org.ametys.cms.repository.comment.Comment;
import org.ametys.cms.repository.comment.CommentsDAO;
import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.cms.transformation.RichTextTransformer;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.user.population.PopulationContextHelper;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.tags.ProjectTagProviderExtensionPoint;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Common helper for workspace objects
 */
public class WorkspaceObjectJSONHelper extends AbstractLogEnabled implements Serviceable
{
    /** Attribute to know if error occurred while handling rich text */
    public static final String ATTRIBUTE_FOR_RICHTEXT_ERROR = "contentError";

    /** The user helper */
    protected UserHelper _userHelper;

    /** The tag provider extension point */
    protected ProjectTagProviderExtensionPoint _tagProviderExtensionPoint;

    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;

    /** The user manager */
    protected UserManager _userManager;

    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;

    /** The population context helper */
    protected PopulationContextHelper _populationContextHelper;

    /** Source resolver */
    protected SourceResolver _sourceResolver;

    /** Rich text transformer */
    protected RichTextTransformer _richTextTransformer;
    /** The service manager */
    protected ServiceManager _smanager;
    
    /** The comments DAO */
    protected CommentsDAO _commentsDAO;

    public void service(ServiceManager manager) throws ServiceException
    {
        _smanager = manager;
        _userHelper = (UserHelper) manager.lookup(UserHelper.ROLE);
        _tagProviderExtensionPoint = (ProjectTagProviderExtensionPoint) manager.lookup(ProjectTagProviderExtensionPoint.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _userManager = (UserManager) manager.lookup(UserManager.ROLE);
        _populationContextHelper = (PopulationContextHelper) manager.lookup(PopulationContextHelper.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _richTextTransformer = (RichTextTransformer) manager.lookup(ThreadDocbookTransformer.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _commentsDAO = (CommentsDAO) manager.lookup(CommentsDAO.ROLE);
    }
    /**
     * return a list of comments in JSON format
     * @param <T> type of the value to retrieve
     * @param comments the comments to translate as JSON
     * @param lang the current language
     * @param siteName The current site name
     * @return list of comments
     */
    protected <T extends AbstractComment> List<Map<String, Object>> _commentsToJson(List<T> comments, String lang, String siteName)
    {
        return _commentsToJson(comments, lang, siteName, null, true);
    }

    /**
     *
     * return a list of comments in JSON format
     * @param <T> type of the value to retrieve
     * @param comments the comments to translate as JSON
     * @param lang the current language
     * @param siteName The current site name
     * @param lastReadDate Last read date to check if the comment is unread
     * @param parseCommentcontent True to parse comment content
     * @return list of comments
     */
    protected <T extends AbstractComment> List<Map<String, Object>> _commentsToJson(List<T> comments, String lang, String siteName, ZonedDateTime lastReadDate, boolean parseCommentcontent)
    {
        List<Map<String, Object>> json = new ArrayList<>();

        for (T comment : comments)
        {
            json.add(_commentToJson(comment, lang, siteName, lastReadDate, parseCommentcontent));
        }

        return json;
    }
    /**
     * Transform the rich text as rendering string
     * @param content the content
     * @return the rendering string
     * @throws IOException if I/O error occurred.
     */
    public String richTextToRendering(RichText content) throws IOException
    {
        Source src = null;
        try (InputStream contentIs = content.getInputStream())
        {

            Map<String, Object> parameters = new HashMap<>();
            parameters.put("source", contentIs);
            parameters.put("dataContext", DataContext.newInstance());
            parameters.put("level", 2);

            src = _sourceResolver.resolveURI("cocoon://plugins/workspaces/convert/docbook2htmlrendering", null, parameters);
            try (InputStream is = src.getInputStream())
            {
                String renderingText = IOUtils.toString(is, "UTF-8");

                // FIXME Try to find a better workaround
                renderingText = StringUtils.replace(renderingText, " xmlns:i18n=\"http://apache.org/cocoon/i18n/2.1\"", "");
                renderingText = StringUtils.replace(renderingText, " xmlns:ametys=\"org.ametys.cms.transformation.xslt.AmetysXSLTHelper\"", "");
                renderingText = StringUtils.replace(renderingText, " xmlns:docbook=\"http://docbook.org/ns/docbook\"", "");
                renderingText = StringUtils.replace(renderingText, " xmlns:project=\"org.ametys.plugins.workspaces.project.helper.ProjectXsltHelper\"", "");

                renderingText = StringUtils.substringAfter(renderingText, "<xml>");
                renderingText = StringUtils.substringBefore(renderingText, "</xml>");
                return renderingText;
            }
        }
        finally
        {
            _sourceResolver.release(src);
        }
    }

    /**
     * Transform the rich text as simple text (without HTML tag)
     * @param richText the content
     * @param maxLength the max length for content excerpt. Set to 0 to not truncate content.
     * @return the simple text, truncated if maxLength > 0
     */
    public String richTextToSimpleText(RichText richText, int maxLength)
    {
        SAXParser saxParser = null;
        try (InputStream is = richText.getInputStream())
        {
            RichTextHandler txtHandler = new RichTextHandler(maxLength);
            saxParser = (SAXParser) _smanager.lookup(SAXParser.ROLE);
            saxParser.parse(new InputSource(is), txtHandler);
            
            return txtHandler.getValue();
        }
        catch (Exception e)
        {
            getLogger().error("Cannot extract simple text from richtext", e);
        }
        finally
        {
            _smanager.release(saxParser);
        }
        
        return "";
    }

    /**
     * return a list of comments in JSON format
     * @param <T> type of the value to retrieve
     * @param comment the comment to translate as JSON
     * @param lang the current language
     * @param siteName The current site name
     * @param lastReadDate Last read date to check if the comment is unread
     * @param parseCommentcontent True to parse comment content
     * @return list of comments
     */
    protected <T extends AbstractComment> Map<String, Object> _commentToJson(T comment, String lang, String siteName, ZonedDateTime lastReadDate, boolean parseCommentcontent)
    {
        Map<String, Object> commentJson = new HashMap<>();
        commentJson.put("id", comment.getId());
        commentJson.put("subComments", _commentsToJson(comment.getSubComment(true, true), lang, siteName));
        commentJson.put("isDeleted", comment.isDeleted());
        commentJson.put("creationDate", comment.getCreationDate());
        commentJson.put("isReported", comment.getReportsCount() > 0);

        if (comment.isSubComment())
        {
            T parent = comment.getCommentParent();
            commentJson.put("parentId", parent.getId());
            User author = _userManager.getUser(parent.getAuthor());
            commentJson.put("parentAuthor", _authorToJSON(parent, author, lang));
        }

        if (!comment.isDeleted())
        {
            commentJson.put("text", comment.getContent());
            
            if (comment instanceof Comment mentionComment)
            {
                
                List<Map<String, Object>> mentionedUsers2json = mentionComment.extractMentions()
                        .stream()
                        .map(user -> _userHelper.user2json(user))
                        .toList();
                commentJson.put("mentions", mentionedUsers2json);
            }
            
            commentJson.put("isEdited", comment.isEdited());
            commentJson.put("nbLike", comment.getReactionUsers(ReactionType.LIKE).size());
            List<String> userLikes = comment.getReactionUsers(ReactionType.LIKE)
                .stream()
                .map(UserIdentity::userIdentityToString)
                .collect(Collectors.toList());
            commentJson.put("userLikes", userLikes);

            commentJson.put("accepted", comment.isAccepted());

            UserIdentity currentUser = _currentUserProvider.getUser();
            commentJson.put("isLiked", userLikes.contains(UserIdentity.userIdentityToString(currentUser)));

            User author = _userManager.getUser(comment.getAuthor());

            commentJson.put("author", _authorToJSON(comment, author, lang));
            commentJson.put("canHandle", author != null ? author.getIdentity().equals(currentUser) : false);
        }

        if (lastReadDate != null && comment.getCreationDate().isAfter(lastReadDate))
        {
            commentJson.put("unread", true);
        }

        if (comment instanceof RichTextComment richTextComment && richTextComment.hasRichTextContent() && parseCommentcontent)
        {

            RichText richText = richTextComment.getRichTextContent();
            StringBuilder result = new StringBuilder(2048);
            try
            {
                _richTextTransformer.transformForEditing(richText, DataContext.newInstance(), result);
                commentJson.put(RichTextComment.ATTRIBUTE_CONTENT_FOR_EDITING, result.toString());
                commentJson.put(RichTextComment.ATTRIBUTE_CONTENT_FOR_RENDERING, richTextToRendering(richText));
            }
            catch (AmetysRepositoryException | IOException e)
            {
                commentJson.put(ATTRIBUTE_FOR_RICHTEXT_ERROR, true);
                getLogger().error("Unable to transform the rich text value into a string", e);
            }

        }
        return commentJson;
    }
    
    /**
     * Author to JSON
     * @param <T> type of the value to retrieve
     * @param comment the comment
     * @param author the author
     * @param lang the current language
     * @return map representing the author
     */
    protected <T extends AbstractComment> Map<String, Object> _authorToJSON(T comment, User author, String lang)
    {
        Map<String, Object> jsonAuthor = new HashMap<>();
        jsonAuthor.put("name", Optional.ofNullable(author).map(User::getFullName).orElse(comment.getAuthorName()));
        if (author != null)
        {
            UserIdentity userIdentity = author.getIdentity();
            jsonAuthor.put("id", UserIdentity.userIdentityToString(userIdentity));
            jsonAuthor.put("login", userIdentity.getLogin());
            jsonAuthor.put("populationId", userIdentity.getPopulationId());
            Content member = _projectMemberManager.getUserContent(lang, author);
            if (member != null)
            {
                if (member.hasValue("function"))
                {
                    jsonAuthor.put("function", member.getValue("function"));
                }
                if (member.hasValue("organisation-accronym"))
                {
                    jsonAuthor.put("organisationAcronym", member.getValue("organisation-accronym"));
                }
            }
        }
        return jsonAuthor;
    }

    /**
     * return a binary in JSON format
     * @param binary the binary
     * @return the binary in JSON format
     */
    protected Map<String, Object> _binaryToJson(Binary binary)
    {
        Map<String, Object> json  = new HashMap<>();

        try (InputStream is = binary.getInputStream())
        {
            json.put("id", binary.getFilename());
            json.put("name", binary.getFilename());
            json.put("type", binary.getMimeType());
            json.put("size", binary.getLength());
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred reading binary {}", binary.getFilename(), e);
        }
        return json;
    }

    /**
     * return a list of tags in JSON format
     * @param taggableAmetysObject the {@link TaggableAmetysObject}
     * @param siteName The current site name
     * @return a list of tags
     */
    protected List<String> _getTags(TaggableAmetysObject taggableAmetysObject, String siteName)
    {
        return taggableAmetysObject.getTags()
                .stream()
                .filter(tag -> _tagProviderExtensionPoint.hasTag(tag, Map.of("siteName", siteName)))
                .collect(Collectors.toList());
    }
}
