/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

try { logger; }
catch (e) { logger = Ametys.getLogger("org.ametys.core.migration.MigrationExtensionPoint.workspaces"); }
  
let projectManager = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.project.ProjectManager.ROLE);
let workspaceModuleExtensionPoint = Ametys.serviceManager.lookup(org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint.ROLE);
let pageDAO = Ametys.serviceManager.lookup(org.ametys.web.repository.page.PageDAO.ROLE);
let observationManager = Ametys.serviceManager.lookup(org.ametys.core.observation.ObservationManager.ROLE);
let currentUserProvider = Ametys.serviceManager.lookup(org.ametys.core.user.CurrentUserProvider.ROLE);

projectManager.getProjects().forEach(function (project) 
{
    let pages = projectManager.getModulePages(project, workspaceModuleExtensionPoint.getModule('org.ametys.plugins.workspaces.wall.WallContentModule'));
    for (let page of pages)
    {
        removeReader(page, true);
        removeReader(page, false);
      
        try 
        {
            let pageLive = Repository.resolver.resolveById(page.getId(), Repository.liveSession);
            removeReader(pageLive, true);
            removeReader(pageLive, false);
        }
        catch (e) {/* No page in live */}
      
        notifyObservers(page);
    }
}); 

function getExistingProfilesForUsers(ametysObject, isAllowed) 
{
    // Compatible for 4.3
    if (ametysObject.getAllowedProfilesForUsers)
    {
       return ametysObject.getAllowedProfilesForUsers().entrySet().iterator();
    }
    else // Compatible for 4.4
    {
    	let userOrGroupEnum = isAllowed ? org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.ALLOWED : org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.DENIED;
        let map = new java.util.HashMap();
        let profiles = ametysObject.getProfilesForUsers(null);
        for (let key of profiles.keySet())
        {
        	map.put(key, profiles.get(key).get(userOrGroupEnum));
        }
       
        return map.entrySet().iterator();
    }
}

function getExistingProfilesForGroups(ametysObject, isAllowed) 
{
    // Compatible for 4.3
    if (ametysObject.getAllowedProfilesForGroups)
    {
        return ametysObject.getAllowedProfilesForGroups().entrySet().iterator();
    }
    else // Compatible for 4.4
    {
    	let userOrGroupEnum = isAllowed ? org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.ALLOWED : org.ametys.core.right.ProfileAssignmentStorage.UserOrGroup.DENIED;
        let map = new java.util.HashMap();
        let profiles = ametysObject.getProfilesForGroups(null);
        for (let key of profiles.keySet())
        {
             map.put(key, profiles.get(key).get(userOrGroupEnum));
        }
       
        return map.entrySet().iterator();
    }
}
  
function removeReader(ametysObject, isAllowed)
{
    // For users
    let existingProfiles = getExistingProfilesForUsers(ametysObject, isAllowed);  
    while (existingProfiles.hasNext())
    {
        let existingProfile = existingProfiles.next();
        if (existingProfile.getValue().contains("READER"))
        {
            logger.info("Removing READER profile for " + existingProfile.getKey());
            if (isAllowed)
            {
                ametysObject.removeAllowedUsers(java.util.Set.of(existingProfile.getKey()), "READER");
            }
            else
            {
                ametysObject.removeDeniedUsers(java.util.Set.of(existingProfile.getKey()), "READER");
            }
        }
    }
    
    // For groups
    existingProfiles = getExistingProfilesForGroups(ametysObject, isAllowed);
    while (existingProfiles.hasNext())
    {
        let existingProfile = existingProfiles.next();
        if (existingProfile.getValue().contains("READER"))
        {
            logger.info("Removing READER profile for " + existingProfile.getKey());
            if (isAllowed)
            {
                ametysObject.removeAllowedGroups(java.util.Set.of(existingProfile.getKey()), "READER");
            }
            else
            {
                ametysObject.removeDeniedGroups(java.util.Set.of(existingProfile.getKey()), "READER");
            }
        }
    }
  
    // For anonymous and for any connected
    let profileIds = new java.util.HashSet();
    profileIds.add("READER");
    if (isAllowed)
    {
        ametysObject.removeAllowedProfilesForAnonymous(profileIds);
        ametysObject.removeAllowedProfilesForAnyConnectedUser(profileIds);
    }
    else
    {
        ametysObject.removeDeniedProfilesForAnonymous(profileIds);
        ametysObject.removeDeniedProfilesForAnyConnectedUser(profileIds);
    }
}
  
function notifyObservers(page)
{
    let profileIds = new java.util.HashSet();
    profileIds.add("READER");
  
    let eventParams = new java.util.HashMap();
    eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT, page);
    eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_CONTEXT_IDENTIFIER, page.getId());
    eventParams.put(org.ametys.core.ObservationConstants.ARGS_ACL_PROFILES, profileIds);

    observationManager.notify(new org.ametys.core.observation.Event(org.ametys.core.ObservationConstants.EVENT_ACL_UPDATED, currentUserProvider.getUser(), eventParams));
}
 