/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

/**
 * This class handles the access to project properties. See {@link Ametys.plugins.workspaces.project.Project}
 */
Ext.define(
	"Ametys.plugins.workspaces.project.ProjectDAO", 
	{
		singleton: true,
		
		/**
		 * Retrieve a sitemap by its id
		 * @param {String} id The project identifier. Cannot be null.
		 * @param {Function} callback The callback function called when the project is retrieved. Can be null for synchronous mode (not recommended!). Parameters are
		 * @param {Ametys.plugins.workspaces.project.Project} callback.project The project retrieved. Can be null if project does not exist.
		 * @return {Ametys.plugins.workspaces.project.Project} The project retrieved if no callback function is specified or null otherwise.
		 */
		getProject: function (id, callback)
		{
			if (Ext.isEmpty(id))
			{
				callback(null);
				return;
			}
			this._sendRequest (id, Ext.bind(this._getProjectCb, this, [callback], 1));
		},
		/**
		 * @private
		 * Callback function called after #getProject is processed
		 * @param {Ametys.plugins.workspaces.project.Project} project The project retrieved
		 * @param {Function} callback The callback function called 
		 */
		_getProjectCb: function (project, callback)
		{
			callback(project);
		},
		
		/**
		 * @private
		 * Send request to server to retrieved project
		 * @param {String[]} id The id of project to retrieve
		 * @param {Function} callback The callback function to be called after server process
		 */
		_sendRequest: function (id, callback)
		{
			Ametys.data.ServerComm.callMethod({
				role: "org.ametys.plugins.workspaces.project.ProjectManager",
				methodName: "getProjectProperties",
				parameters: [id],
				callback: {
					scope: this,
					handler: this._getProjectPropertiesCb,
					arguments: [callback, id]
				},
				errorMessage: {
					category: this.self.getName(),
					msg: "{{i18n PLUGINS_WORKSPACES_DAOS_PROJECT_ERROR}}"
				}
			});
		},
		
		/**
		 * @private
		 * Callback function called after #_sendRequest is processed
		 * @param {Object} data The project properties
		 * @param {Array} args The callback arguments.
		 */
		_getProjectPropertiesCb: function (data, args)
		{
			if (!data)
			{
				Ametys.log.ErrorDialog.display({
					title: "{{i18n PLUGINS_WORKSPACES_DAOS_PROJECT_NOT_FOUND}}", 
					text: "{{i18n PLUGINS_WORKSPACES_DAOS_PROJECT_NOT_FOUND_HINT}}",
		    		details: args[1],
		    		category: "Ametys.plugins.workspaces.project.ProjectDAO"
				});
			}
            else
            {
                var sitemap = Ext.create ('Ametys.plugins.workspaces.project.Project', data);

                if (typeof args[0] == 'function')
                {
                	args[0] (sitemap);
                }
            }
		}
	}
);
