/*
 *  Copyright 2022 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
ActivityStream = {
    
    FILTER_SELECTOR: '[data-ametys-activitystream-role=filters]',
    
    RESULTS_SELECTOR: '[data-ametys-activitystream-role=results]',
    
    NORESULT_SELECTOR: '[data-ametys-activitystream-role=no-result]',
    
    RESULTSTATUS_SELECTOR: '[data-ametys-activitystream-role=results-status]',
    
    SHOW_MORE_SELECTOR: '[data-ametys-activitystream-role=show-more]',
    
    ERROR_MSG_SELECTOR: '[data-ametys-activitystream-role=error-msg]',
        
    // The site uri prefix
    SITE_URI_PREFIX: null,
    // the site name
    SITE_NAME: null,
    // the sitemap language
    LANG: null,
    
    // refresh delay in minutes
    _refreshDelay: 3,
    
    // max number of events retrieved on each server call
    _maxEvents: 100,
    
    // jquery selector
    _selector : null,
    
    // last update date
    _lastUpdate: null,
    
    // date of last displayed activity
    _lastActivityDate: null,
    
    // is activity stream currently open ?
    isOpened: false,
    
    // is activity stream out-of-date ?
    outOfDate: false,
    
    // are activities should be mark as read ?
    markAsRead: true,
    
    /**
     * Configure the activity stream
     * @param {Number} refreshDelay the refresh delay in minutes to get unread events. Default to 3.
     * @param {Number} maxEvents the max number of events to display. Default to 100.
     */
    configure: function(siteUriPrefix, siteName, lang, refreshDelay, maxEvents)
    {
        ActivityStream.SITE_URI_PREFIX = siteUriPrefix;
        ActivityStream.SITE_NAME = siteName;
        ActivityStream.LANG = lang;
        
        this._refreshDelay = refreshDelay || 3;
        this._maxEvents = maxEvents || 100;
    },
    
    /**
     * Whatever the activities should be marked as read at next loading
     * @param {} markAsRead Set to true to mark activities as read at next loading. Set to false otherwise.
     */
    setMarkAsRead: function(markAsRead)
    {
         ActivityStream.markAsRead = markAsRead;
    },
    
    /**
     * Active the ping to get the number of unread events
     * @param {String} [selector] a jquery selector
	 * @param {String} [projectName] the project name to filter unread events
     */
    getNumberOfUnreadEvents: function(selector, projectName)
    {
        if (ActivityStream._unreadTimeOut)
        {
            clearTimeout(ActivityStream._unreadTimeOut);
            ActivityStream._unreadTimeOut = null;
        }
        
        let $selector = $j(selector);
        
        $j.ajax({
            url: ActivityStream.SITE_URI_PREFIX + "/_plugins/workspaces/activity-stream/nb-unread-events.json",
            type: 'POST',
            dataType: 'json',
			data: projectName ? { 'projectName': projectName } : {},
            success: function(data, status, error) 
            {
                $selector.find(ActivityStream.ERROR_MSG_SELECTOR).hide();
                
                if (data.nbUnreadEvents != 0)
                {
                    // activity stream is out of date
                    ActivityStream.outOfDate = true;
                    AmetysFront.Event.fire("activity-stream-outofdate", {nbUnreadEvents: data.nbUnreadEvents});
                }
                else
                {
                    AmetysFront.Event.fire("activity-stream-uptodate");
                }
                
                // Get number of unread events every N minutes
                ActivityStream._unreadTimeOut = setTimeout(function() {
                    ActivityStream.getNumberOfUnreadEvents(selector, projectName);
                }, ActivityStream._refreshDelay * 60000);
            },
            
            error: function(result, status, error) {
                console.error("Unable to get number of unread events. Status is " + result.status + ".");
                $selector.find(ActivityStream.ERROR_MSG_SELECTOR).show();
            }
        });
    },
    
    /**
     * Open and load activity stream if necessary
     * @param {String} selector a jquery selector
     * @param {Function} [callback] a callback function to call after loading
     * @param {Boolean} [force=false] set to true to force reload even if the activity is not mark as outofdate
     * @param {Boolean} [append=false] set to true append to results
     */
    open: function(selector, callback, force, append)
    {
        if (!this._lastUpdate)
        {
            // First initalization: add listeners
            AmetysFront.Event.listen("activity-stream-opened", function(data) {
                ActivityStream.isOpened = true;
	        })
	         
	        AmetysFront.Event.listen("activity-stream-closed", function(data) {
	           ActivityStream.isOpened = false;
	        })
            
            AmetysFront.Event.listen("activity-stream-outofdate", function(data) {
                // On out of date reload activity stream only if it is opened or visible
                if (ActivityStream.isOpened)
                {
                    ActivityStream.open(selector, callback, true, false);
                 }
            })
        }
        
        let now = new Date().getTime();
        
        if (!ActivityStream._lastUpdate || ActivityStream.outOfDate || force)
        {
            ActivityStream._lastUpdate = now;
            
            let $selector = $j(selector);
            
            AmetysFront.Event.fire("activity-stream-loading");
            
            let data = {
                'max-results': this._maxEvents,
                'mark-as-read': ActivityStream.markAsRead
            };
            
            this._handleFilters($selector, data);
            
            if (append && ActivityStream._lastActivityDate)
            {
                // get next former activities
                data['untilDate'] = this._lastActivityDate
            }
            
            var me = this;
            $j.ajax({
                url: ActivityStream.SITE_URI_PREFIX + "/_plugins/workspaces/" + ActivityStream.SITE_NAME + "/" + ActivityStream.LANG + "/activity-stream.xml",
                type: 'POST',
                dataType: 'html',
                data: data,
                traditional: true,
                success: function(data, status, error) 
                {
                    $selector.find(ActivityStream.ERROR_MSG_SELECTOR).hide();
                    
                    var tmpDoc = document.implementation.createHTMLDocument('');
                    tmpDoc.documentElement.innerHTML = data;
                    
                    var $content = $j(tmpDoc).find('body');
                    
                    var $tmp = $j('<div style="display: none; position: absolute; top: -10000">').appendTo('body');
                    $tmp.html($content.html());
                    
                    var nbResults = $tmp.find("ul.ametys-notifications__list > li").length;
                    
                    var $results = $selector.find(ActivityStream.RESULTS_SELECTOR);
                    
                    if (!append)
                    {
                        // replace
                        $results.html($tmp.find("ul.ametys-notifications__list"));
                    }
                    else
                    {
                        // append after
                        var $firstNewResult = $tmp.find(">ul.ametys-notifications__list > li").first();
                        $tmp.find("ul.ametys-notifications__list > li").appendTo($results.find(">ul.ametys-notifications__list"));
                        $firstNewResult.attr("tabindex", -1);
                        $firstNewResult.trigger('focus');
                    }
                    
                    nbResults == 0 ? $selector.find(ActivityStream.SHOW_MORE_SELECTOR).hide() : $selector.find(ActivityStream.SHOW_MORE_SELECTOR).show();
                    
                    var nbVisibleResults = $results.find("ul.ametys-notifications__list > li").length;
                    nbVisibleResults == 0 ? $selector.find(ActivityStream.NORESULT_SELECTOR).show() : $selector.find(ActivityStream.NORESULT_SELECTOR).hide();
                    
                    if (!append && force)
                    {
                        // RGAA purpose: update a live zone with role="status" when activity stream is reloaded after a search filter
                        if (nbVisibleResults == 0)
                        {
                            $selector.find(ActivityStream.RESULTSTATUS_SELECTOR).html("{{i18n PLUGINS_WORKSPACES_SERVICE_ACTITY_STREAM_STATUS_SEARCH_NORESULT}}");
                        }
                        else
                        {
                            // Number of results displayed after executing filter
                            $selector.find(ActivityStream.RESULTSTATUS_SELECTOR).html(nbVisibleResults + "{{i18n PLUGINS_WORKSPACES_SERVICE_ACTITY_STREAM_STATUS_SEARCH_RESULTS}}");
                        }
                    }
                    
                    $tmp.remove();
                    $j(tmpDoc).remove();
                    
                    ActivityStream._lastActivityDate = nbResults > 0 ? $results.find("> ul.ametys-notifications__list > li:last-child [data-momentdate]").data('momentdate') : ActivityStream._lastActivityDate;
                    
                    ActivityStream.updateEventDates(selector);
                    
                    if (typeof callback == 'function')
                    {
                        callback();
                    }
                    
                    if (!append)
                    {
                        AmetysFront.Event.fire("activity-stream-uptodate");
                    }
                    
                    AmetysFront.Event.fire("activity-stream-loaded", {force: force, append: append});
                },
                
                error: function(result, status, error) {
                    console.error("Unable to get activity stream. Status is " + result.status + ".");
                    
                    $selector.find(ActivityStream.ERROR_MSG_SELECTOR).show();
                }
            });
        }
    },
    
    _handleFilters: function($selector, params)
    {
        var $filter = $selector.find(ActivityStream.FILTER_SELECTOR);
        
        $filter.serializeArray()
            .forEach(function(param) {
                var name = param.name,
                val = param.value;
	            if (name && params.hasOwnProperty(name) && val)
	            {
	                let valuesArray = $j.makeArray(params[name]);
	                valuesArray.push(val);
                    params[name] = valuesArray;
	            }
	            else if (name && val)
	            {
	                params[name] = val;
	            }
            });
    },
    
    updateEventDates: function(selector)
    {
        if (ActivityStream._updateDatesTimeout)
        {
            clearTimeout(ActivityStream._updateDatesTimeout);
            ActivityStream._updateDatesTimeout = null;
        }
        
        $j(selector).find("span[data-momentdate]").each(function() {
            let eventDate = $j(this).data('momentdate');
            $j(this).html(AmetysFront.Utils.toRelativeDate(eventDate, {withTime: true}));
        });
        
        // refresh events' date every 1 min
        ActivityStream._updateDatesTimeout = setTimeout(function() { 
            ActivityStream.updateEventDates(selector)
        }, 60000);
    }
}