/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
AmetysWorkspacePreferences = {
    /**
     * Read the notification form and send the preferences to the server.
     * @param uniqueId the unique Id of the form
     */
    updateUserPrefs: function (uniqueId) {
        var form = $j('#preference-default-notification-form-' + uniqueId);
        var prefs = {};
        prefs.disable = form.find('input[name="disable"]:checked').length === 1;
        prefs.frequency = form.find('input[name="frequency"]:checked').val();
        prefs.projects = {};
        form.find('input[type="hidden"]').each(function ()
        {
            var input = $(this); 
            prefs.projects[input.attr("name")] = JSON.parse($(this).val());
        });
        AmetysFront.Event.fire('loaderStart', {
            text: "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_SERVICE_PREFERENCE_NOTIFICATIONS_SERVER_COMM_WAITING_TEXT}}"
        });
        AmetysFront.ServerComm.send({
            plugin: "web",
            url: "service/user-preferences.html",
            responseType: 'xml',
            parameters: {'submit': "true",
                         'siteName': AmetysFront.getAppParameter("siteName"),
                         'lang': AmetysFront.getAppParameter("lang"),
                         'preferences-ids': ['worspaces.notifications'],
                         'workspaces.notifications': JSON.stringify(prefs),
                         'ajax': true,
                         },
            callback: {
                handler: AmetysWorkspacePreferences._updateUserPrefsCb,
                scope: this,
                arguments: []
            }
        });
    },
    
    /**
     * @private
     * Check that userpref were updated and display feedback to the user
     */
    _updateUserPrefsCb: function (response) {
        if (response != null)
        {
            // we should tell the user about the success
             AmetysFront.Event.fire('loaderEnd', {
                  text: "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_SERVICE_PREFERENCE_NOTIFICATIONS_SERVER_COMM_SUCCESS_TEXT}}"
                });
        }
        else
        {
            // something went wrong, we should tell the user
            AmetysFront.Event.fire('loaderFail', {
                  title: "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_SERVICE_PREFERENCE_NOTIFICATIONS_SERVER_COMM_TITLE}}",
                  text: "{{i18n plugin.workspaces:PLUGINS_WORKSPACES_SERVICE_PREFERENCE_NOTIFICATIONS_SERVER_COMM_FAILURE_TEXT}}"
            });
        }
    },
    
    /**
     * @private
     * Listener to enable/disable other input based on the disable input
     */
    _toggleDisableListener: function (event)
    {
        AmetysWorkspacePreferences._toggleDisable($j(event.delegateTarget));
    },
    
    /**
     * @private
     * enable/disable other input based on the disable input
     */
    _toggleDisable: function (form)
    {
        var elemSelectorAr = [];
        elemSelectorAr.push('[data-notification-frequency]');
        elemSelectorAr.push('[data-notification-project-prefs]');
        var elemSelector = elemSelectorAr.join(', ');
        if (form.find('[name="disable"]').is(':checked'))
        {
            form.find('[name="frequency"], [data-notification-project-prefs] [data-action]').attr('disabled', true);
            form.find(elemSelector).addClass('disabled');
        }
        else
        {
            form.find('[name="frequency"], [data-notification-project-prefs] [data-action]').attr('disabled', false);
            form.find(elemSelector).removeClass('disabled');
        }
    },
    
    /**
     * initialize the service based on the provided prefs value and project list.
     * The method also attach the required listener.
     * 
     * @param {String} uniqueId the uniqueId of the service
     * @param {Object} prefs the current prefs value
     * @param {Object} projects the list of project with their name and title
     */
    populateService: function (uniqueId, prefs, projects) {
        var form = $j(`#preference-default-notification-form-${uniqueId}`);
        form.find(`input[name="disable"]`).prop("checked", prefs.disable);
        AmetysWorkspacePreferences._toggleDisable(form);
        form.find(`input[name="frequency"][value="${prefs.frequency}"]`).prop("checked", true);
		
		var projectSelector = $j('#preference-new-project-notification-modal-' + uniqueId + ' [name="project-name"]');
		
        // Consider all projects as without custom preferences and add them to the selector
        for (var projectName in projects)
        {
            AmetysWorkspacePreferences._addProjectOptionToSelector(projectSelector, projectName, projects[projectName].title, uniqueId);
        }
        
        // Remove customized project from selector and add them to the list of customized
        for (var projectName in prefs.projects)
        {
            // we check that the user still has access to the project
            // if not, we ignore the pref and it will be remove at the next save from the user pref
            if (projects[projectName] != null)
            {
                AmetysWorkspacePreferences._removeProjectOptionFromSelector(projectSelector, projectName, uniqueId);
                AmetysWorkspacePreferences._addProjectPrefsInput(form, projectName, prefs.projects[projectName]);
                AmetysWorkspacePreferences._addProjectItem(projectName, projects[projectName].title, uniqueId);
            }
        }
        
        form.on('click', '[data-action="open-new-project-modal"]', function(event) {AmetysWorkspacePreferences._setupNewProjectModal(event, uniqueId)});
        
        form.find('[data-notification-project-prefs]').on('click', '.ametys-more:not(.disabled)', function() {$('.ametys-more').not($(this)).removeClass('active'); $(this).toggleClass('active');});
        form.on('click', 'a[data-action="open-edit-modal"]', function(event) {AmetysWorkspacePreferences._setupProjectModal(event, uniqueId)});
        form.on('click', 'a[data-action="delete-project-prefs"]', function(event) {AmetysWorkspacePreferences._removeProjectUserPrefs(event, uniqueId)});
        
        form.on('change', 'input[name="disable"]', AmetysWorkspacePreferences._toggleDisableListener);
        form.on('change', function(_event) {AmetysWorkspacePreferences.updateUserPrefs(uniqueId)});
        
        var newProjectModal = $j('#preference-new-project-notification-modal-' + uniqueId);
        newProjectModal.on('click', 'button[class="modal-apply"]', function(event) {AmetysWorkspacePreferences._updateProjectUserPrefs(event, uniqueId)});
        newProjectModal.on('change', 'input[name="disable"]', AmetysWorkspacePreferences._toggleDisableListener);
        
        var projectModal = $j('#preference-project-notification-modal-' + uniqueId);
        projectModal.on('click', 'button[class="modal-apply"]', function(event) {AmetysWorkspacePreferences._updateProjectUserPrefs(event, uniqueId)});
        projectModal.on('change', 'input[name="disable"]', AmetysWorkspacePreferences._toggleDisableListener);
    },
    
    /**
     * @private
     * Add an hidden input to the form with the preferences value
     * @param form the form element
     * @param {String} projectName the name of the project
     * @param {Object} projectPrefs the preferences of the project
     */
    _addProjectPrefsInput: function(form, projectName, projectPrefs)
    {
        $j('<input type="hidden">').attr({
            name: projectName,
            value: JSON.stringify(projectPrefs)
        }).appendTo(form);
    },
    
    /**
     * @private
     * Remove the project input from the form
     * @param form the form element
     * @param {String} projectName the name of the project
     */
    _removeProjectPrefsInput: function(form, projectName)
    {
        // here we match both attribute to avoid collision with default prefs
        form.find('input[name=' + projectName + '][type="hidden"]').remove();
    },
    
    /**
     * @private
     * Update the project preferences
     * @param event the event triggering the method
     * @param uniqueId the unique Id of the service
     */
    _updateProjectUserPrefs: function (event, uniqueId)
    {
        var modal = $j(event.delegateTarget);
        
        var form = modal.find('form');
        var projectNameField = form.find('[name="project-name"]');
        var projectName = projectNameField.val();
        if (projectName != '')
        {
            var projectPrefs = {};
            projectPrefs.disable = form.find('input[name="disable"]:checked').length === 1;
            projectPrefs.frequency = form.find('input[name="frequency"]:checked').val();
            
            var defaultForm = $j('#preference-default-notification-form-' + uniqueId);
            var projectInput = defaultForm.find('input[type="hidden"][name=' + projectName + ']');
            if (projectInput.length < 1)
            {
                var projectTitle = projectNameField.find('option[value="' + projectName + '"]').text();
                AmetysWorkspacePreferences._addProjectPrefsInput(defaultForm, projectName, projectPrefs)
                AmetysWorkspacePreferences._addProjectItem(projectName, projectTitle, uniqueId);
				var projectSelector = $j('#preference-new-project-notification-modal-' + uniqueId + ' [name="project-name"]');
                AmetysWorkspacePreferences._removeProjectOptionFromSelector(projectSelector, projectName, uniqueId);
            }
            else
            {
                projectInput.attr("value", JSON.stringify(projectPrefs));
            }
            defaultForm.trigger('change')
            modal.modal('hide');
        }
    },
    
    /**
     * @private
     * Remove project specific preferences
     * @param event the event triggering the listener
     * @param uniqueId the unique id of the service
     */
    _removeProjectUserPrefs: function (event, uniqueId)
    {
        var currentTarget = $j(event.currentTarget);
        var defaultForm = $j('#preference-default-notification-form-' + uniqueId);
        
        var projectName = currentTarget.attr('data-project-name');
        AmetysWorkspacePreferences._removeProjectPrefsInput(defaultForm, projectName);
        AmetysWorkspacePreferences.updateUserPrefs(uniqueId);
		
		var projectSelector = $j('#preference-new-project-notification-modal-' + uniqueId + ' [name="project-name"]');
        AmetysWorkspacePreferences._addProjectOptionToSelector(projectSelector, projectName, currentTarget.attr('data-project-title'), uniqueId);
        var newProjectButton = defaultForm.find('[data-action="open-new-project-modal"]');
        if (newProjectButton.is(':hidden') == true)
        {
            newProjectButton.show();
        }
        currentTarget.parents('[data-project-prefs-item]').remove();
    },
    
    /**
     * @private
     * Add the project option to the selector for new project specific preferences
	 * @param {Object} selector the selector jQuery object
     * @param {String} projectName the project name
     * @param {String} projectTitle the title of the project
     * @param {String} uniqueId the unique id of the service
     */
    _addProjectOptionToSelector: function (selector, projectName, projectTitle, uniqueId)
    {
        $j('<option>').attr({value: projectName, 'data-tokens': projectName})
                      .text(projectTitle)
                      .appendTo(selector);
    },
    
    /**
     * @private
     * Remove the project option from the selector for new project specific preferences
	 * @param {Object} selector the selector jQuery object
     * @param {String} projectName the project name
     * @param {String} uniqueId the unique id of the service
     */
    _removeProjectOptionFromSelector: function (selector, projectName, uniqueId)
    {
        selector.find(`option[value= ${projectName}]`).remove();
        if (selector.find('option[data-tokens]').length == 0)
        {
            $j('#preference-default-notification-form-' + uniqueId + ' [data-action="open-new-project-modal"]').hide();
        }
    },
    
    /**
     * @private
     * Add the project item in the list of item with specific preferences
     * @param {String} projectName the project name
     * @param {String} projectTitle the project title'
     * @param {String} uniqueId the unique id of the service
     */
    _addProjectItem: function (projectName, projectTitle, uniqueId)
    {
        let data = {name: projectName, title: projectTitle};

        let projectItemTemplate = $j(`#project-item-template`)
        let projectItemElmt = $j(document.createElement("div"));
        projectItemElmt.hide();
        $j('body').append(projectItemElmt);
        
        projectItemElmt.html(projectItemTemplate.html());
        
        $j.each(data, function(key, value) {
            let elmts = projectItemElmt.find("*[data-project-" + key + "]")
            
            elmts.each(function(_i, elmt) {
                let attr = $(elmt).data('project-' + key);
                
                if (attr)
                {
                    $(elmt).attr(attr, value);
                }
                else
                {
                    $(elmt).append(value);
                    $(elmt).show();
                }
            })
        })
        
        $j(`#preference-default-notification-form-${uniqueId} [data-notification-projects-list]`).append(projectItemElmt.children());
        projectItemElmt.remove();

    },
    
    /**
     * @private
     * Edit the modal to match the project properties
     * @param event the event triggering the listener
     * @param {String} uniqueId the unique id of the service
     */
    _setupProjectModal: function (event, uniqueId)
    {
        var currentTarget = $j(event.currentTarget);
        var projectName = currentTarget.attr("data-project-name");
        var projectTitle = currentTarget.attr("data-project-title");
        if (projectTitle === "")
        {
            projectTitle = currentTarget.text()
        }
        
        var projectPrefs = JSON.parse($j('#preference-default-notification-form-' + uniqueId + ' input[type="hidden"][name="' + projectName + '"]').val());
        var modal = $j('#preference-project-notification-modal-' + uniqueId);
        var title = modal.find(' .modal-title');
        title.empty();
        title.append(projectTitle);
        
        modal.find(`[name="project-name"]`).val(projectName).attr('data-project-title', projectTitle);
        modal.find(`[name="disable"]`).prop("checked", projectPrefs.disable).trigger('change');
        if (projectPrefs.frequency != '')
        {
            modal.find(`[name="frequency"][value=${projectPrefs.frequency}]`).prop("checked", true);
        }
        
        modal.modal('show');
    },
    
    /**
     * @private
     * Edit the modal to match the default properties
     * @param _event the event triggering the listener
     * @param {String} uniqueId the unique id of the service
     */
    _setupNewProjectModal: function(_event, uniqueId)
    {
        var modal = $j('#preference-new-project-notification-modal-' + uniqueId);
        var defaultForm = $j('#preference-default-notification-form-' + uniqueId);
        modal.find('[name=disable]').prop("checked", false).trigger('change');
        modal.find('[name=frequency][value="' + defaultForm.find('input[name="frequency"]:checked').val() + '"]').prop("checked", true);
        modal.modal('show');
		
		// Refresh the project selector in case options were added/removed
		var projectSelector = $j('#preference-new-project-notification-modal-' + uniqueId + ' [name="project-name"]').selectpicker("refresh");
		projectSelector.selectpicker("refresh");
    }
}
