/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */


/*
 * Composable used to handle dates, like formatting or comparing them.
 */

/**
 * Formats a date to include the short weekday and day number.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted date with short weekday and day number.
 * @example
 * getDayNumberWithWeekday(new Date("2025-05-14")); // "14 Wed" with "en" as locale
 * getDayNumberWithWeekday(new Date("2025-05-14")); // "mer. 14" with "fr" as locale
 */
export function getDayNumberWithWeekday(date) {
    return date.toLocaleString(AmetysFront.getAppParameter("locale"), {weekday: 'short', day: 'numeric' })
}

/**
 * Formats a date to a long readable string including weekday, month, and day.
 * @param {Date|string} date - The date to format.
 * @returns {string} - The formatted date with weekday, day and month.
 * @example
 * getDateWithWeekdayAndMonth("2025-05-14"); // "Wednesday, May 14" with "en" as locale
 * getDateWithWeekdayAndMonth("2025-05-14"); // "mercredi 14 mai" with "fr" as locale
 */
export function getDateWithWeekdayAndMonth(date)
{
    let locale = AmetysFront.getAppParameter("locale");

    let options = {
      weekday: "long",
      month: "long",
      day: "numeric",
    }
    
    return new Date(date).toLocaleString(locale, options);
}

/**
 * Formats a date to include only the day number.
 * @param {Date} date - The date to format.
 * @param {Object} [config] - Optional configuration for locale and formatting.
 * @returns {string} - The formatted day number .
 * @example
 * getDayNumber(new Date("2025-05-14")); // "14"
 * getDayNumber(new Date("2025-05-04")); // "04"
 * getDayNumber(new Date("2025-05-04", {day: "numeric"})); // "4"
 */
export function getDayNumber(date, config)
{
    config = config || {};

    let locale = config.locale || AmetysFront.getAppParameter("locale");

    let dateConfig = {
        "day": config.day || "2-digit", 
    };
    return new Date(date).toLocaleString(locale, dateConfig);
}

/**
 * Formats a date to include the short month name.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted short month.
 * @example
 * getShortMonth(new Date("2025-05-14")); // "May" with "en" as locale
 * getShortMonth(new Date("2025-05-14")); // "mai" with "fr" as locale
 * getShortMonth(new Date("2025-09-14")); // "Sep" with "en" as locale
 * getShortMonth(new Date("2025-09-14")); // "sept." with "fr" as locale
 */
export function getShortMonth(date)
{
    let locale = AmetysFront.getAppParameter("locale");

    let dateConfig = {
        "month": "short", 
    };
    return new Date(date).toLocaleString(locale, dateConfig);
}

/**
 * Converts a date to the "HH:mm" format.
 * @param {Date} originalDate - The date to format.
 * @returns {string} - The formatted time.
 * @example
 * getTimeInHHmmFormat(new Date("2025-05-14T14:00:00")); // "14:00"
 */
export function getTimeInHHmmFormat(originalDate)
{
    var date = new Date(originalDate);
    
    const hour = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    return hour + ":" + minutes;
}

/**
 * Converts a date to a readable time string in "HH:mm" format.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted time.
 * @example
 * getReadableTime(new Date("2025-05-14T14:00:00")); // "02:00 PM" with "en" as locale
 * getReadableTime(new Date("2025-05-14T14:00:00")); // "14:00" with "fr" as locale
 */
export function getReadableTime(date)
{
    let locale = AmetysFront.getAppParameter("locale");
    
    let dateConfig = {
        "hour": "2-digit",
        "minute": "2-digit",
    };

    return new Date(date).toLocaleString(locale, dateConfig);
}

/**
 * Converts a date to a readable time string in "HH:mm" format.
 * @param {Date} date - The date to format.
 * @param {Object} prefixMap - An object containing prefixes for different time formats.
 * @param {string} prefixMap.defaultPrefix - The default prefix for the formatted time less than three hours ago.
 * @param {string} prefixMap.hoursPrefix - The prefix for the formatted time more than three hours ago but today.
 * @param {string} prefixMap.dayPrefix - The prefix for the formatted time for days in the current week.
 * @param {string} prefixMap.datePrefix - The prefix for the formatted time for dates older than a week.
 * @returns {string} - The formatted time relative to today, with a prefix.
 * @example
 * With the current date being "2025-05-14T14:30:00":
 * getRelativeDateWithPrefix(new Date("2025-05-14T14:00:00")); // "30 minutes ago" with "en" as locale
 * getRelativeDateWithPrefix(new Date("2025-05-14T14:00:00")); // "il y a 25 minutes" with "fr" as locale
 * getRelativeDateWithPrefix(new Date("2025-05-08T14:00:00")); // "jeudi" with "fr" as locale
 * getRelativeDateWithPrefix(new Date("2025-04-11T14:00:00")); // "le 11 mars" with "fr" as locale
 */
export function getRelativeDateWithPrefix(date, prefixMap)
{
    let text = prefixMap.defaultPrefix;
    
    let d1 = new Date();
    d1.setHours(0, 0, 0, 0); // at midnight
    
    let d2 = new Date(date);
    d2.setHours(0, 0, 0, 0); // at midnight
    
    let diffDates = (d1 - d2) / (24 * 60 * 60 * 1000);
    
    let timeDiff = (new Date() - new Date(date)) / 1000;
    
    // If the date is today but more than three hours ago, we display the hour/minutes so we need a prefix
    if (timeDiff >= 3*3600 && diffDates == 0)
    {
        text = prefixMap.hoursPrefix;
    }
    else if (diffDates > 1 && diffDates < 7)
    {
        text = prefixMap.dayPrefix
    }
    // If the difference between the two dates is greater than 7 days, we display the date using toLocaleString so we need a prefix
    else if (diffDates >= 7)
    {
        text = prefixMap.datePrefix;
    }

    text = text.replace(/\{0\}/, AmetysFront.Utils.toRelativeDate(date, {withTime: true}));
    return text;
}

/**
 * Converts a date to the "YYYY-MM-DD" format, taking.
 * @param {Date} originalDate - The date to format.
 * @returns {string} - The formatted date.
 * @example
 * dateToTimePickerFormat(new Date("2025-05-14")); // "2025-05-14"
 */
export function dateToTimePickerFormat(originalDate)
{
    var date = new Date(originalDate);
    
    // Take timezone into account, so that we get the correct day
    // As the date is in ISO format is UTC, it may be one day off if hour is early/late
    const offset = date.getTimezoneOffset()
    var dateWithOffset = new Date(date.getTime() - (offset*60*1000))
    return dateWithOffset.toISOString().split('T')[0];
}

/**
 * Converts a date to a local ISO string with timezone offset.
 * @param {Date} originalDate - The date to format.
 * @returns {string} - The formatted ISO string .
 * @example
 * dateToLocalISO(new Date()); // "'2025-05-14T14:10:04.259+02:00'"
 */
export function dateToLocalISO(originalDate) {
    
    var date = new Date(originalDate);
    const off = date.getTimezoneOffset()
    const absoff = Math.abs(off)
    return (new Date(date.getTime() - off*60*1000).toISOString().substr(0,23) +
            (off > 0 ? '-' : '+') + 
            Math.floor(absoff / 60).toFixed(0).padStart(2,'0') + ':' + 
            (absoff % 60).toString().padStart(2,'0'))
}

/**
 * Formats a date to a readable string in "day month year hour:minute" format.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted date.
 * @example
 * dateToLLLFormat(new Date("2025-05-14T14:00:00")); // "May 14, 2025 at 02:00 PM" with "en" as locale
 * dateToLLLFormat(new Date("2025-05-14T14:00:00")); // "14 mai 2025 à  14:00" with "fr" as locale
 */
export function dateToLLLFormat(date) {
      let config = {
              "day": "numeric", 
              "month": "long", 
              "year": "numeric",
              "day": 'numeric',
              "hour": '2-digit',
              "minute": '2-digit',
          };

      let locale = AmetysFront.getAppParameter("locale");

      return new Date(date).toLocaleString(locale, config);
}

/**
 * Formats a date to a readable string in "day short month year" format.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted date.
 * @example
 * dateTollFormat(new Date("2025-05-14")); // "14 May 2025" with "en" as locale
 * dateTollFormat(new Date("2025-05-14")); // "14 mai 2025" with "fr" as locale
 * dateTollFormat(new Date("2025-09-09")); // "Sep 9, 2025" with "en" as locale
 * dateTollFormat(new Date("2025-09-09")); // "9 sept. 2025" with "fr" as locale
 */
export function dateTollFormat(date) {
    return AmetysFront.Utils.toReadableDate(date, {
          "day": "numeric", 
          "month": "short", 
          "year": "numeric"
      })
}

/**
 * Formats a date to a readable string in "month year" format.
 * @param {Date} date - The date to format.
 * @returns {string} - The formatted date.
 * @example
 * dateToMonthYearFormat(new Date("2025-05-14")); // "May 2025" with "en" as locale
 * dateToMonthYearFormat(new Date("2025-05-14")); // "mai 2025" with "fr" as locale
 */
export function dateToMonthYearFormat(date) {
    let locale = AmetysFront.getAppParameter("locale");
    
    let options = {
      "month": "long",
      "year": "numeric",
    }
    
    return new Date(date).toLocaleString(locale, options);
}

/**
* Formats a date to a readable string in "day month year" format.
* @param {Date} date - The date to format.
* @returns {string} - The formatted date.
* @example
* dateToDayMonthYearFormat(new Date("2025-05-14")); // "May 14, 2025" with "en" as locale
* dateToDayMonthYearFormat(new Date("2025-05-14")); // "14 mai 2025" with "fr" as locale
*/
export function dateToDayMonthYearFormat(date) {
    let locale = AmetysFront.getAppParameter("locale");
    
    let options = {
      "month": "long",
      "year": "numeric",
      "day": "numeric", 
    }
    
    return new Date(date).toLocaleString(locale, options);
}

/**
 * Compares two dates by day only and checks if the first date is after the second.
 * @param {Date} date1 - The first date.
 * @param {Date} date2 - The second date.
 * @returns {boolean} - True if `date1` is after `date2` (ignoring time), false otherwise.
 * @example
 * isAfterComparingOnlyDays(new Date("2025-05-14"), new Date("2025-05-13")); // true
 */
export function isAfterComparingOnlyDays(date1, date2)
{
    const date1Day = new Date(date1);
    date1Day.setHours(0, 0, 0, 0);
    
    const date2Day = new Date(date2);
    date2Day.setHours(0, 0, 0, 0);
    
    return date1Day > date2Day;
}

/**
 * Checks if a new start date, adjusted by the difference from the original start date, is after the until date.
 * @param {Date} originalStartDate - The original start date.
 * @param {Date} newStart - The new start date.
 * @param {Date} originalOccurrenceStart - The original occurrence start date.
 * @param {Date} untilDate - The until date to compare against.
 * @returns {boolean} - True if the adjusted occurrence is after `untilDate`, false otherwise.
 * @example
 * isNewDateAfterUntilDate(new Date("2025-05-10"), new Date("2025-05-14"), new Date("2025-05-09"), new Date("2025-05-20")); // false
 * isNewDateAfterUntilDate(new Date("2025-05-10"), new Date("2025-06-14"), new Date("2025-05-09"), new Date("2025-05-20")); // true
 */
export function isNewDateAfterUntilDate (originalStartDate, newStart, originalOccurrenceStart, untilDate)
{
    const diffMs = newStart - originalStartDate;
    
    const adjustedOccurrence = new Date(originalOccurrenceStart.getTime() + diffMs);
    
    return adjustedOccurrence > untilDate;
}

/**
 * Adds years to the given date.
 * @param {Date} date - The date to modify.
 * @param {Number} yearsToAdd - The number of years to add.
 * @returns {Date} - The new date with N year added.
 * @example
 * addYears(new Date("2025-05-14"), 1); // "2026-05-14"
 */
export function addYears(date, yearsToAdd) {
    const newDate = new Date(date);
    newDate.setYear(newDate.getFullYear() + yearsToAdd);
    return newDate;
}

/**
 * Adds days to the given date.
 * @param {Date} date - The date to modify.
 * @param {Number} daysToAdd - The number of days to add.
 * @returns {Date} - The new date with N days added.
 * @example
 * addDays(new Date("2025-05-14"), 1); // "2025-05-15"
 * addDays(new Date("2025-05-14"), -1); // "2025-05-13"
 */
export function addDays(date, daysToAdd) {
    const newDate = new Date(date);
    newDate.setDate(newDate.getDate() + daysToAdd);
    return newDate;
}


/**
 * Adds hours to the given date.
 * @param {Date} date - The date to modify.
 * @param {Number} hoursToAdd - The number of hours to add.
 * @returns {Date} - The new date with N hours added.
 * @example
 * addDays(new Date("2025-05-14T14:00:00"), 1); // "2025-05-14T15:00:00"
 */
export function addHours(date, hoursToAdd) {
    const newDate = new Date(date);
    newDate.setHours(newDate.getHours() + hoursToAdd);
    return newDate;
}