/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';
import { useEventFiltersStore } from '@/stores/eventFilters'
import { useEventsStore } from '@/stores/events'

/**
 * Pinia store for handling calendars
 */
export const useCalendarsStore = defineStore('calendars', {
    state: () => (
    {
        calendars: [], // list of calendars
        type: null, // type of calendar to display ('calendar' or 'resource')
        firstTimeTaskCalendarIsEnabled: true
    }),
    getters: 
    {
        /**
         * Get the list of calendars with their colors
         */
        getCalendars: state =>
        {
            const data = [];

            state.calendars.forEach((el) => {
                const color = state.$colorsMap[el.color] || state.$.colorsMap["col1"];
                data.push({ ...el, color: color});
            });

            return data;
        },
        
        /**
         * Get the list of active calendars with their colors
         */
        getActiveCalendars: state =>
        {
            const data = [];

            state.calendars.forEach((el) => {
                if (!el.isTaskCalendar || !el.isTaskCalendarDisabled)
                {
                    const color = state.$colorsMap[el.color] || state.$.colorsMap["col1"];
                    data.push({ ...el, color: color});
                }
            });

            return data;
        },
        
        /**
         * Get the list of modifiable calendars with their colors
         */
        getModifiableCalendars: state =>
        {
            const data = [];

            state.calendars.forEach((el) => {
                if (!el.isTaskCalendar)
                {
                    const color = state.$colorsMap[el.color] || state.$.colorsMap["col1"];
                    data.push({ ...el, color: color});
                }
            });

            return data;
        },
        
        /**
         * get the current calendar type
         */
        getCalendarType: state =>
        {
            if (state.type) return state.type;

            const filtersStore = useEventFiltersStore();
            return filtersStore.filterByCalendar.resource ? 'resource' : 'calendar';
        },

        /**
         * Get the resource calendar with its color
         */
        getResourceCalendar: state =>
        {
            const color = state.$colorsMap[state.$resourceCalendar.color] || state.$colorsMap["col1"];
            return {...state.$resourceCalendar, color: color};
        },
    },
    actions:
    {
        /**
         * Set the type of calendar to display
         * @param {String} type the type of calendar ('calendar' or 'resource')
         */
        setType(type)
        {
            this.type = type;
        },
        
        /**
         * Add a new calendar
         * @param {Object} calendar the calendar object to add
         * @param {String} calendar.title the title of the calendar
         * @param {Object} calendar.color the color object of the calendar
         * @param {Boolean} calendar.public whether the calendar is public or not
         */
        async addCalendar(calendar)
        {
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_CALENDAR_BEGIN_MSG
            });
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.CalendarDAO',
                methodName: 'addCalendar',
                parameters: [
                    calendar.title,
                    calendar.color.id,
                    calendar.public
                ],
            })
            .then((data) => {
                this.calendars.push(data);
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_CALENDAR_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_ADD_CALENDAR_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
        
        /**
         * Update an existing calendar
         * @param {Object} calendar the calendar to update
         * @param {String} calendar.id the id of the calendar
         * @param {String} calendar.title the title of the calendar
         * @param {String} calendar.templateDesc the description of the calendar
         * @param {Object} calendar.color the color object of the calendar
         * @param {Boolean} calendar.public - Whether the calendar is public or not
         */
        async updateCalendar(calendar)
        {
            const eventStore = useEventsStore();
            const filtersStore = useEventFiltersStore();

            var that = this;
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_CALENDAR_BEGIN_MSG
            });
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.CalendarDAO',
                methodName: calendar.isTaskCalendar ? 'editTaskCalendar' : 'editCalendar',
                parameters: calendar.isTaskCalendar 
                    ? [calendar.title, calendar.color.id, calendar.public, calendar.isTaskCalendarDisabled]
                    : [calendar.id, calendar.title, calendar.templateDesc, calendar.color.id, calendar.public]
            })
            .then((data) => {
                
                if (calendar.isTaskCalendar && !calendar.isTaskCalendarDisabled && this.firstTimeTaskCalendarIsEnabled)
                {
                    eventStore.loadEvents(filtersStore.start, filtersStore.end);
                    var start = new Date().toISOString();
                    var end = new Date();
                    end.setFullYear(end.getFullYear() + 1);
                    end = end.toISOString();
                    eventStore.loadAsideEvents(start, end);
                    this.firstTimeTaskCalendarIsEnabled = false;
                }
                else
                {
                    const color = that.$colorsMap[data.color] ? data.color : "col1";
                    eventStore.updateEventCalendar({
                        id: data.id,
                        title: data.title,
                        public: data.visibility == "PUBLIC",
                        color: color,
                        isTaskCalendar: data.isTaskCalendar,
                        isTaskCalendarDisabled: data.isTaskCalendarDisabled
                    });
                }
                this.calendars = this.calendars.map(calendar => calendar.id === data.id ? data : calendar);
               
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_CALENDAR_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_EDIT_CALENDAR_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
        
        /**
         * Delete a calendar
         * @param {String} calendarId the id of the calendar to delete
         */
        async deleteCalendar(calendarId)
        {
            const eventStore = useEventsStore();
            AmetysFront.Event.fire('loaderStart', {
                text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_CALENDAR_BEGIN_MSG
            });
           
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.CalendarDAO',
                methodName: 'deleteCalendar',
                parameters: [calendarId],
            })
            .then(() => {
                this.calendars = this.calendars.filter(calendar => calendar.id != calendarId);
                eventStore.removeCalendarFromEvents(calendarId);
                AmetysFront.Event.fire('loaderEnd', {
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_CALENDAR_END_MSG
                });
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_DELETE_CALENDAR_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
        
        /**
         * Load the list of calendars
         */
        async loadCalendars()
        {
            await callMethod({
                role: 'org.ametys.plugins.workspaces.calendars.CalendarDAO',
                methodName: 'getCalendars',
            })
            .then(data => {
                this.calendars = data;
                this.calendars = this.calendars.filter(calendar => calendar.id != this.$resourceCalendar.id);
                this.calendars.forEach(calendar => {
                    if (calendar.isTaskCalendar)
                    {
                        if (!calendar.title)
                        {
                            calendar.title = i18n.PLUGINS_WORKSPACES_CALENDAR_TASK_DEFAULT_TITLE;
                        }

                        if (!calendar.isTaskCalendarDisabled)
                        {
                            this.firstTimeTaskCalendarIsEnabled = false;            
                        }
                    }
                });
                
                
            })
            .catch((error) => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_LOAD_CALENDAR_ERROR_MSG,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_CALENDAR_GENERAL_ERROR_TEXT,
                    details: error,
                });
            });
        },
    }
})

