/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import { useCalendarsStore } from '@/stores/calendars'

/**
 * Pinia store for filtering events
 */
export const useEventFiltersStore = defineStore('eventFilters', {
    state: () => (
    {
        filterByName: '', // The name to filter events by (not used in the current implementation)
        filterByCalendar: false, // The calendar to filter events by
        filterByResourceCalendar: false, // True to filter only resource calendars
        selectedCalendar: {}, // The selected calendar
        start: null, // The start date of the range to filter events
        end: null, // The end date of the range to filter events
    }),
    getters:
    {
        /**
         * Check if a filter is applied
         * @return {boolean} true if a filter is applied, false otherwise
         */
        isFilter: (state) =>
        {
            return state.filterByName.length > 0 || Object.keys(state.filterByCalendar).length > 0;
        },
    },
    actions:
    {
        /**
         * Set the date range for the events to be loaded
         * @param {Date} start the start date of the range
         * @param {Date} end the end date of the range
         */
        setDateRange(start, end)
        {
            this.start = start;
            this.end = end;
        },
        
        /**
         * Set the name filter for the events
         * @param {string} value the name to filter by
         */
        setByName(value)
        {
            this.filterByName = value;
        },
        
        /**
         * Set the category filter for the events
         * @param {Object} value the category object to filter by
         */
        setByCalendar(category)
        {
            this.filterByCalendar = category;
            this.filterByResourceCalendar = false;
    
            this.selectedCalendar = {
              id: category.id,
              color: category.color,
              icon: category.icon,
              name: category.name || category.title,
              title: category.title,
            };
        },
        
        /**
         * Reset he current calendar filter
         */
        resetCategory()
        {
            this.filterByCalendar = false;
            this.selectedCalendar = {};
        },
        
        /**
         * filter by resource
         * @param {boolean} value true to filter by resource calendar, false otherwise
         */
        setByResourceCalendar(value)
        {
            this.filterByResourceCalendar = value;
    
            this.filterByCalendar = false;
    
            this.selectedCalendar = {};
        },
        
        /**
         * Reset all filters
         */
        resetFilters()
        {
            const calendarsStore = useCalendarsStore();
            this.setByName('');
            this.setByCalendar(false);

            calendarsStore.setType(null);
        },
    }
})

