/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';
import i18n from 'i18n';


/**
 * Pinia store for handling comments in files
 */
export const useCommentStore = defineStore('comment', {
    state: () => ({
          comments: {} // The comments for the current file
      }),
    actions:
    {
      /**
       * Load comments
       * @param {Object} file the file for which to load comments
       */
      async loadComments(file) {
    
        this.comments = [];
    
        try
        {
            let comments = await callMethod({
                role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceCommentsDAO',
                methodName: 'getComments',
                parameters: [ file.id]
            });

            this.comments = comments;
        }
        catch(e)
        {
            AmetysFront.Event.fire('loaderFail', {
                title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_FILE_ERROR_COMMENTS,
                text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_COMMENTS_GET_ERROR,
                details: null
            });
        }
      },
    
      /**
       * Add new comment
       * @param {string} ametysObjectId the id of the Ametys object to comment
       * @param {string} text the text of the comment as it is stored
       */
      async addComment(ametysObjectId, text) {
        var that = this;
        return new Promise(function(resolve, reject)
        {
            callMethod({
                role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceCommentsDAO',
                methodName: 'addComment',
                parameters: [ ametysObjectId, text]
            })
            .then(comments => {
                that.comments = comments;
                resolve({comments: comments});
            })
            .catch(error => {
                AmetysFront.Event.fire('loaderFail', {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_COMMENTS_ADD_ERROR,
                    text : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: error
                });
    
                reject();
            });
        });
      }
  }
})
