/*
 *  Copyright 2019 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

import { defineStore } from 'pinia'
import { callMethod } from '@common/helper/ServerCommHelper'
import AmetysFront from 'AmetysFront'
import i18n from 'i18n'
import { useFoldersStore } from '@/stores/folders'
import { useFilesStore } from '@/stores/files'


/**
 * Pinia store for managing selected items (folders and files).
 */
export const useSelectedItemsStore = defineStore('selectedItems', {
    state: () => (
    {
        items: [], // All selected items
        folders: [], // Selected folders
        files: [], // Selected files
        isSelectAll: false // Whether "selectAll" is active
    }),
    getters:
    {
        /**
         * Returns the total number of selected items.
         */
        getSelectedCount: (state) =>
        {
            return state.files.length + state.folders.length
        },
        /**
         * Checks if an item is selected.
         */
        getIsSelectedItem: (state) => (payload) =>
        {
            if (state[payload.type].length > 0)
            {
                return !!state[payload.type].find(item => item.id === payload.id)
            }
            else
            {
                return false
            }
        },
        /**
         * Checks if any selected item can be deleted.
         */
        canDeleteItems: (state) =>
        {
            let canDelete = false
            state.files.forEach(f =>
            {
                if (f.rights['delete'])
                {
                    canDelete = true
                    return
                }
            })
            state.folders.forEach(f =>
            {
                if (f.rights['delete'])
                {
                    canDelete = true
                    return
                }
            })
            return canDelete
        }
    },
    actions:
    {
        /**
         * Selects all folders and files.
         */
        selectAll()
        {
            const foldersStore = useFoldersStore()
            const filesStore = useFilesStore()
            const folders = foldersStore.getFolders || []
            const files = filesStore.getFiles || []
            this.folders = folders
            this.files = files
            this.isSelectAll = true
        },
        /**
         * Unselects all items and fires an event for each.
         */
        unselectAll()
        {
            let count = this.getSelectedCount
            this.folders = []
            this.files = []
            this.isSelectAll = false
            while (count > 0)
            {
                AmetysFront.Event.fire('showMobileHeader')
                count--
            }
        },
        /**
         * Selects a single item.
         * @param {string} type the type of the item ('folders' or 'files')
         * @param {Object} item the item to select
         */
        select(type, item)
        {
            const found = this[type].filter(f => f.id == item.id)
            if (found)
            {
                this[type].push(item)
            }
            AmetysFront.Event.fire('hideMobileHeader')
        },

        /**
         * Unselects a single item.
         * @param {string} type the type of the item ('folders' or 'files')
         * @param {Object} item the item to unselect
         */
        unselect(type, item)
        {
            if (this.getIsSelectedItem({ type: type, id: item.id }))
            {
                this[type] = this[type].filter(value => value.id !== item.id)
                this.isSelectAll = false
            }
            AmetysFront.Event.fire('showMobileHeader')
        },

        /**
         * Deletes selected items.
         */
        async deleteSelectedItems()
        {
            const foldersStore = useFoldersStore()
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETING
            })
            let itemIds = []
            this.files.forEach(f =>
            {
                if (f.rights['delete'])
                {
                    itemIds.push(f.id)
                }
            })
            this.folders.forEach(f =>
            {
                if (f.rights['delete'])
                {
                    itemIds.push(f.id)
                }
            })
            try
            {
                let result = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO',
                    methodName: 'deleteObject',
                    parameters: [itemIds]
                })
                if (!result.error)
                {
                    foldersStore.openFolder(foldersStore.currentFolder)
                    this.folders = []
                    this.files = []
                    this.isSelectAll = false
                    setTimeout(() =>
                    {
                        AmetysFront.Event.fire('loaderEnd',
                        {
                            text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_ITEMS_SUCCESS
                        })
                    }, 500)
                    foldersStore.loadStorageSpace();
                }
                else
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_ITEMS_ERROR,
                        text: result.error == 'locked'
                            ? i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_ITEMS_ERROR_LOCKED
                            : i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                        details: null
                    })
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_DELETE_ITEMS_ERROR,
                    text: i18n.PLUGINS_WORKSPACES_FILE_MANAGER_ERROR,
                    details: null
                })
            }
        },

        /**
         * Downloads selected items as a zip archive.
         */
        downloadSelectedItems()
        {
            let params = "name=export"
            this.folders.forEach(f => { params += "&file=" + f.id })
            this.files.forEach(f => { params += "&file=" + f.id })
            window.location.href = AmetysFront.getPluginDirectPrefix('explorer') + '/folder/archive.zip?' + params
        }
    }
})
