/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
import { defineStore } from 'pinia'
import AmetysFront from 'AmetysFront';
import { callMethod } from '@common/helper/ServerCommHelper.js';

import { useTagsStore } from '@common/store/project-tags/tags'
import i18n from 'i18n';

const resultsPerPage = 15;

/**
 * Pinia store for handling threads
 */
export const useThreadStore = defineStore('thread',
{
    state: () => (
    {
        userRights: {}, // current user rights
        filters: {}, // map of thread filters
        tags: [], // list of available tags
        usedTags: [], // list of used used tags
        threadsPagination: {}, // pagination for infinite scroll
        categoryThreadsCount: null, // map of thread categories and their threads count
        threadIds: [], // list of thread ids, used internally for infinite scroll and contains all threads ids
        threads: [], // list of loaded threads, filled by infinite scroll
        thread: {}, // current thread
        initialThreadCategory: '' // initial thread category of the current thread, used to update the categoryThreadsCount if category is changed
    }),
    actions:
    {
        /**
         * Load the rights of the current user
         */
        async loadUserRights()
        {
            AmetysFront.Event.fire('loader', true);
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getUserRights',
                })
                .then(data =>
                {
                    AmetysFront.Event.fire('loader', false);
                    this.userRights = data;
                })
                .catch(() =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_RIGHTS_ERROR_MSG,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                    });
                });
        },

        /**
         * Get threads ids list
         * @param {String} q  filter by search query
         * @param {String} category  filter by categroy id
         * @param {Number} tag  filter by tag id
         * @param {Number} closed  filter by closeInfo field
         * @param {Boolean} accepted  filter by whether threads have accepted answer
         * @param {Boolean} hasNotification  filter by whether threads have notification
         */
        async getThreadIds({q, category, tag, closed, accepted, hasNotification})
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS
            });
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getThreadIds',
                    parameters: [q || null, category || null, tag || null, closed, accepted, hasNotification],
                })
                .then(threadIds =>
                {
                    AmetysFront.Event.fire('loaderEnd');
                    this.threadIds = threadIds;
                })
                .catch((error) =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        details: error
                    });
                });
        },

        /**
         * Get threads list by page number
         * @param {Number} page the number of the page to load
         */
        async getThreadsByPage(page)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS
            });

            var threadIds = this.threadIds.slice((page - 1) * resultsPerPage, resultsPerPage * page);
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getThreadsByIds',
                    parameters: [threadIds],
                })
                .then(threads =>
                {
                    AmetysFront.Event.fire('loaderEnd');

                    this.threadsPagination = {
                        "perPage": resultsPerPage,
                        "page": page,
                        "lastPage": this.threadIds.length == 0 ? 1 : Math.ceil(this.threadIds.length / resultsPerPage),
                        "total": this.threadIds.length,
                    };
                    
                    if (page == 1)
                    {
                        this.threads = threads;
                    }
                    else
                    {
                        this.threads = [...this.threads, ...threads];
                    }
                })
                .catch((error) =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        details: error
                    });
                });
        },

        /**
         * Get last thread of the page, used when a thread is deleted for example
         * @param {Number} page the number of the page to load
         */
        async getLastThread(page)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS
            });

            var threadIds = this.threadIds.slice(resultsPerPage * page - 1, resultsPerPage * page);
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getThreadsByIds',
                    parameters: [threadIds],
                })
                .then(threads =>
                {
                    AmetysFront.Event.fire('loaderEnd');
                    if (threads[0] != null)
                    {
                        this.threadsPagination = {
                            "perPage": resultsPerPage,
                            "page": page,
                            "lastPage": this.threadIds.length == 0 ? 1 : Math.ceil(this.threadIds.length / resultsPerPage),
                            "total": this.threadIds.length,
                        };
                        this.threads = [...this.threads, ...threads];
                    }
                })
                .catch((error) =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREADS_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        details: error
                    });
                });
        },

        /**
         * Get detailed threads to view
         * @param {Number} id  id of the requested thread
         */
        async getThread(id)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREAD
            });
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getThread',
                    parameters: [id],
                })
                .then(data =>
                {
                    AmetysFront.Event.fire('loaderEnd');
                    if (data)
                    {
                        this.thread = data;
                        this.initialThreadCategory = data.category;
                    }
                })
                .catch((error) =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_THREAD_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        details: error
                    });
                });
        },

        /**
         * Create a new thread
         * @param {Object} thread The thread parameters
         * @param {Object} query The search query, used to know if we display the created thread
         * @param {String} query.q filter by search query
         * @param {String} query.category filter by category id
         * @param {String} query.tag filter by tag id
         * @param {Boolean} query.closed filter by closeInfo field
         * @param {Boolean} query.accepted filter by accepted answer
         * @param {Boolean} query.hasNotification filter by whether subjects have notification
         * @param {Object[]} newFiles the new file to add
         * @param {String[]} newFileNames the file names to add
         */
        async createThread({thread, query, newFiles, newFileNames})
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_CREATING_THREAD
            });
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'addThread',
                    parameters: [thread, query.q, query.category, query.tag, query.closed, newFiles, newFileNames, query.accepted, query.hasNotification],
                })
                .then(data =>
                {
                    const tagsStore = useTagsStore();
                    tagsStore.addTags(data.newTags);
                    if (data.thread.passFilter)
                    {
                        // Add the new thread on the top
                        this.threads.unshift(data.thread);
                        this.threadIds.unshift(data.thread.id);
                        // If there are still threads in the server, remove the last to keep same amount of threads
                        if (this.threadsPagination.page != this.threadsPagination.lastPage)
                        {
                          // If there are still threads in the server, remove the last to keep same amount of threads
                          if (this.threadsPagination.page != this.threadsPagination.lastPage)
                          {
                            this.threads.pop();
                          }
                        }
                    }
                    if (data.thread.category) {
                        this.categoryThreadsCount[data.thread.category] = (this.categoryThreadsCount[data.thread.category] || 0) + 1;
                    }
                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_CREATED
                    });
                })
                .catch((error) =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_CREATING_THREAD_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        details: error
                    });
                });
        },

        /**
         * Init the thread during thread creation
         * @param {String} category the thread category
         */
        async initThread(category)
        {
            this.thread = {
              category: category,
              tags: [],
              attachments: [],
              isAuthor: true,
            };
        },

        /**
         * Edit a thread
         * @param {Object} thread The thread parameters
         * @param {Object} query The search query, used to know if we display the created thread
         * @param {String} query.q filter by search query
         * @param {String} query.category filter by category id
         * @param {String} query.tag filter by tag id
         * @param {Boolean} query.closed filter by closeInfo field
         * @param {Boolean} query.accepted filter by accepted answer
         * @param {Boolean} query.hasNotification filter by whether subjects have notification
         * @param {Object[]} newFiles list of files to add
         * @param {String[]} newFileNames list of file names to add
         * @param {String[]} deleteFiles list of names of old files to delete
         */
        async updateThread({thread, query, newFiles, newFileNames, deleteFiles})
        {
            return new Promise((resolve, reject) =>
            {

                AmetysFront.Event.fire('loaderStart',
                {
                    text: i18n.PLUGINS_WORKSPACES_FORUM_EDITING_THREAD
                });
                callMethod(
                    {
                        role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                        methodName: 'editThread',
                        parameters: [thread, query.q, query.category, query.tag, query.closed, newFiles, newFileNames, deleteFiles, query.accepted, query.hasNotification],
                    })
                    .then(data =>
                    {
                        AmetysFront.Event.fire('loaderEnd',
                        {
                            text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_EDITED
                        });
                        
                        this.updateThreadInState(data.thread);

                        const tagsStore = useTagsStore();
                        tagsStore.addTags(data.newTags);
                        
                        this.initialThreadCategory = data.thread.category ? data.thread.category : '';
                        resolve(
                        {
                            thread: data.thread
                        });
                    })
                    .catch(() =>
                    {
                        AmetysFront.Event.fire('loaderFail',
                        {
                            title: i18n.PLUGINS_WORKSPACES_FORUM_EDITING_THREAD_ERROR_TITLE,
                            text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                        });
                        reject();
                    });
            });

        },

        /**
         * Delete thread
         * @param {Int} id the id of the thread
         */
        async deleteThread(id)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_DELETING_THREAD
            });
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'deleteThread',
                    parameters: [id],
                })
                .then(thread =>
                {
                    AmetysFront.Event.fire('loaderEnd',
                    {
                        text: i18n.PLUGINS_WORKSPACES_FORUM_THREAD_DELETED
                    });
                    if (thread.id)
                    {
                        this.thread = null;
                    }
                    this.threads = this.threads.filter(s => s.id != thread.id);
                    this.threadIds = this.threadIds.filter(threadId => threadId != thread.id);
                    this.categoryThreadsCount[thread.category] = this.categoryThreadsCount[thread.category] > 0 ? this.categoryThreadsCount[thread.category] - 1 : 0;
                })
                .catch(() =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_DELETING_THREAD_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                    });
                });
        },

        /**
         * Get the number of threads for a given category
         * @param {String} id the id of the category
         */
        async getCategoryThreadCount(id)
        {
            AmetysFront.Event.fire('loaderStart',
            {
                text: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_CATEGORY_THREAD_COUNT
            });
            await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: 'getCategoryThreadCount',
                    parameters: [id],
                })
                .then((count) =>
                {
                    AmetysFront.Event.fire('loaderEnd');

                    if (this.categoryThreadsCount == null) 
                    {
                        this.categoryThreadsCount = { [id]: count };
                    }
                    else
                    {
                        this.categoryThreadsCount[id] = count 
                    }
                })
                .catch(() =>
                {
                    AmetysFront.Event.fire('loaderFail',
                    {
                        title: i18n.PLUGINS_WORKSPACES_FORUM_LOADING_CATEGORY_THREAD_COUNT_ERROR_TITLE,
                        text: i18n.PLUGINS_WORKSPACES_FORUM_GENERAL_ERROR_TEXT,
                    });
                });
        },

        /**
         * Load the tags used in any thread
         */
        async loadUsedTags()
        {
            try
            {
                let tags = await callMethod(
                {
                    role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO',
                    methodName: "getUsedTags",
                    parameters: []
                })

                tags.sort(function(c1, c2)
                {
                    if (AmetysFront.Utils.deemphasize(c1.title.toLowerCase()) == AmetysFront.Utils.deemphasize(c2.title.toLowerCase())) return 0;
                    else if (AmetysFront.Utils.deemphasize(c1.title.toLowerCase()) < AmetysFront.Utils.deemphasize(c2.title.toLowerCase())) return -1;
                    else return 1;
                });

                this.usedTags = tags;
                if (this.filters.tag && !tags.map(tag => tag.name).includes(this.filters.tag))
                {
                  delete this.filters.tag;
                }
            }
            catch (e)
            {
                AmetysFront.Event.fire('loaderFail',
                {
                    title: i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL,
                    text: i18n.PLUGINS_WORKSPACES_PROJECT_LOADTAGS_FAIL_TEXT,
                    details: e
                });

            }
        },

        /**
         * Clear notification
         * @param {String} id id of the thread
         * @param {Boolean} filterByNotification true if we filter by whether threads have any notification, 
         */
        async clearUnopenedThreadNotification(id, filterByNotification)
        {
            await callMethod(
            {
                role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadUserPreferencesDAO',
                methodName: 'clearUnopenedThreadNotification',
                parameters: [id],
            }).then(() =>
            {
                (this.threads || []).forEach((thread) => {
                  if (thread.id == id)
                  {
                    thread.hasUnopenedThreadNotification = false;
                    thread.unreadComments = 0;
                  }
                });

                if (this.thread && this.thread.id == id)
                {
                    this.thread.hasUnopenedThreadNotification = false;
                    this.thread.unreadComments = 0;
                }

                // if we filter by notifications, remove thread from threads list as the thread does not have a notification anymore
                if (filterByNotification)
                {
                    this.threads = this.threads.filter(s=> s.id != id);
                }
            });
        },

        /**
         * Clear unread comments notifications
         * @param {Object} thread  the thread
         * @param {Boolean} filterByNotification filter by whether threads have any notification
         * @param {String[]} commentIds list of comments to clear notifications for
         */
        async clearUnreadCommentsNotification(thread, filterByNotification, commentIds)
        {

            let lastSeenCommentId = commentIds[commentIds.length - 1];
            await callMethod(
            {
                role: 'org.ametys.plugins.workspaces.forum.WorkspaceThreadUserPreferencesDAO',
                methodName: 'clearUnreadCommentsNotification',
                parameters: [thread.id, lastSeenCommentId],
            }).then(() =>
            {
                let lastCommentId = thread.comments[thread.comments.length - 1].id;
                
                this.threads.forEach((t) => 
                {
                  if (t.id == thread.id)
                  {
                    t.unreadComments = t.unreadComments - commentIds.length;
                    if (lastCommentId == lastSeenCommentId)
                    {
                        thread.hasUnopenedThreadNotification = false;
                    }
                  }
                });

                if (this.thread && this.thread.id == thread.id) {
                    this.thread.unreadComments = this.thread.unreadComments - commentIds.length;
                    this.thread.comments.forEach(c=>
                    {
                        if(commentIds.includes(c.id)) c.unread = false;
                    })
                    if (lastCommentId == lastSeenCommentId)
                    {
                        this.thread.hasUnopenedThreadNotification = false;
                        
                        // if we filter by notifications, remove thread from threads list as the thread does not have a notification anymore
                        if (filterByNotification)
                        {
                          this.threads = state.threads.filter(s=> s.id != thread.id);
                        }
                    }
                }
                
            });
        },

        /**
         * Update the thread in the state, and handle the category counts, filters, tags and threads list
         * Used only internally, as several sources can update the thread (update fields, but also all comments interactions)
         * @param {Object} thread the thread to update
         */
        async updateThreadInState(thread)
        {
            this.thread = thread;
            // update category counts if catageory changes
            const previous = this.initialThreadCategory;
            const newCategory = thread.category;
            const counts = this.categoryThreadsCount;
            if (previous != newCategory && counts && counts[previous] != undefined) {
              this.categoryThreadsCount[previous] = this.categoryThreadsCount[previous] > 0 ? this.categoryThreadsCount[previous] - 1 : 0;
              this.categoryThreadsCount[newCategory] = (this.categoryThreadsCount[newCategory] || 0) + 1;
            }

            // update tags this if new/updated thread has new tags
            let newTags = [];
            thread.tags.forEach((tag) => {
              const found = this.tags.find((t) => t.id == tag.id);
              if (!found) newTags.push(tag);
            });

            if (newTags.length > 0) this.tags = [...this.tags, ...newTags];

            // update threads state
            // If thread still pass filters, update the threads to take changes into account
            if (thread.passFilter)
            {
              this.threads = this.threads.map(u => u.id !== thread.id ? u : thread);
            }
            // If thread does not pass filter anymore, remove it from list of threads
            else
            {
              this.threads = this.threads.filter(s=> s.id != thread.id);
            }
        },
    }
})