/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.activitystream;

import java.time.ZonedDateTime;
import java.util.Arrays;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.WebHelper;

/**
 * Component to return a map for action to get the activities for activity stream
 */
public class ActivityStreamServiceComponent implements Serviceable, Component, Contextualizable
{
    /** The avalon role */
    public static final String ROLE = ActivityStreamServiceComponent.class.getName();
    
    private ActivityStreamClientInteraction _activityStream;
    private ProjectManager _projectManager;
    private UserPreferencesManager _userPrefManager;
    private CurrentUserProvider _currentUserProvider;
    private WorkspaceModuleExtensionPoint _moduleEP;
    private Context _context;

    
    public void service(ServiceManager manager) throws ServiceException
    {
        _activityStream = (ActivityStreamClientInteraction) manager.lookup(ActivityStreamClientInteraction.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _userPrefManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _moduleEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    /**
     * Get the activity stream of the current user. In project context, only the activities of the current project are returned, unless allActivities is true.
     * @param limit The max number of results
     * @param allActivities Set to true to force to get activities from all projects of the current user, even if we are in a project context
     * @param markAsRead If true mark the activities as read
     * @return The activity stream
     * @throws UserPreferencesException If an error occurs while marking activities as read
     */
    public Map<String, Object> getActivityStream(int limit, boolean allActivities, boolean markAsRead) throws UserPreferencesException
    {
        Request request = ContextHelper.getRequest(_context);
        
        // The filtered modules
        Set<String> activityTypes = _getFilteredActivityTypes(request);
        
        // The filtered projects' categories
        Set<String> categories = _getFilteredCategories(request);

        // The filtered projects
        Set<String> projects = _getFilteredProjects(request);

        // Fultext search
        String pattern = request.getParameter("pattern");
        
        // Dante range
        ZonedDateTime fromDate = null;
        String fromDateAsStr = request.getParameter("fromDate");
        if (StringUtils.isNotEmpty(fromDateAsStr))
        {
            fromDate = DateUtils.parseZonedDateTime(fromDateAsStr);
        }
        ZonedDateTime untilDate = null;
        String untilDateAsStr = request.getParameter("untilDate");
        if (StringUtils.isNotEmpty(untilDateAsStr))
        {
            untilDate = DateUtils.parseZonedDateTime(untilDateAsStr);
        }
        
        String siteName = WebHelper.getSiteName(request);
        
        List<String> currentProjectNames = _projectManager.getProjectsForSite(siteName);
        Project currentProject = !currentProjectNames.isEmpty() ? _projectManager.getProject(currentProjectNames.get(0)) : null;

        List<Map<String, Object>> activities;
        if (currentProject == null || allActivities)
        {
            // Get activity stream of all current user's projects
            activities = _activityStream.getActivitiesForCurrentUser(pattern, projects, categories, activityTypes, fromDate, untilDate, limit);
        }
        else
        {
            // Get activity stream  of current project only
            activities = _activityStream.getActivitiesForCurrentUser(Collections.singleton(currentProject), activityTypes, fromDate, untilDate, pattern, limit);
        }
        
        UserIdentity user = _currentUserProvider.getUser();
        if (markAsRead && user != null)
        {
            // Mark as read in user prefs
            String userPrefContext = ActivityStreamClientInteraction.ACTIVITY_STREAM_USER_PREF_CONTEXT;
            if (currentProject != null && !allActivities)
            {
                // if we are in a project, use its name as context
                userPrefContext += "/" + currentProject.getName();
            }
            _userPrefManager.setUserPreferences(user, userPrefContext, Map.of(), Map.of(ActivityStreamClientInteraction.ACTIVITY_STREAM_USER_PREF_LAST_UPDATE, DateUtils.dateToString(new Date())));
        }
        
        Map<String, Object> result = new HashMap<>();
        // FIXME this should be rename activities
        result.put("events", activities);
        
        return result;
    }
    
    private Set<String> _getFilteredActivityTypes(Request request)
    {
        String[] moduleIds = request.getParameterValues("modules");
        if (moduleIds != null)
        {
            return Arrays.stream(moduleIds)
                    .filter(StringUtils::isNotEmpty)
                    .map(id -> _moduleEP.<WorkspaceModule>getModule(id))
                    .map(WorkspaceModule::getAllowedEventTypes)
                    .flatMap(Set::stream)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
    
    private Set<String> _getFilteredCategories(Request request)
    {
        String[] categoryIds = request.getParameterValues("categories");
        if (categoryIds != null)
        {
            return Arrays.stream(categoryIds)
                    .filter(StringUtils::isNotEmpty)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
    
    private Set<String> _getFilteredProjects(Request request)
    {
        String[] projectIds = request.getParameterValues("projects");
        if (projectIds != null)
        {
            return Arrays.stream(projectIds)
                    .filter(StringUtils::isNotEmpty)
                    .collect(Collectors.toSet());
        }
        else
        {
            return Set.of();
        }
    }
}
