/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.forums;

import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;

import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.language.UserLanguagesManager;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.forum.Thread;
import org.ametys.plugins.workspaces.forum.WorkspaceThreadDAO;
import org.ametys.runtime.config.Config;

/**
 * Class representing a activity notifier for reporting workspaces thread comment
 */
public class ReportThreadCommentActivityNotifier extends ThreadsActivityNotifier
{
    /** The user languages manager */
    protected UserLanguagesManager _userLanguageManager;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userLanguageManager = (UserLanguagesManager) manager.lookup(UserLanguagesManager.ROLE);
    }
    
    @Override
    public boolean supports(ActivityType activityType)
    {
        return activityType instanceof ThreadCommentReportedActivityType;
    }
    
    @Override
    public Map<String, List<String>> getUsersToNotifyByLanguage(Activity activity)
    {
        boolean returnAll = Config.getInstance().getValue("runtime.mail.massive.sending");

        Thread thread = (Thread) getTargetAmetysObject(activity);
        AmetysObject project = thread.getParent();

        Set<UserIdentity> users = _rightManager.getAllowedUsers(WorkspaceThreadDAO.RIGHTS_REPORT_NOTIFICATION, project).resolveAllowedUsers(returnAll);

        Set<UserIdentity> usersWithOwnThreadRights = _rightManager.getAllowedUsers(WorkspaceThreadDAO.RIGHTS_REPORT_NOTIFICATION_OWN_THREAD, project)
                .resolveAllowedUsers(returnAll)
                .stream()
                .filter(user -> user.equals(thread.getAuthor())).collect(Collectors.toSet());
        
        users.addAll(usersWithOwnThreadRights);
        
        String defaultLanguage = _userLanguageManager.getDefaultLanguage();
        
        return users.stream()
            .map(_userManager::getUser)
            .filter(Objects::nonNull)
            .map(user -> Pair.of(user.getLanguage(), user.getEmail()))
            .filter(p -> StringUtils.isNotEmpty(p.getRight()))
            .collect(Collectors.groupingBy(
                    p -> {
                        return StringUtils.defaultIfBlank(p.getLeft(), defaultLanguage);
                    },
                    Collectors.mapping(
                            Pair::getRight,
                            Collectors.toList()
                    )
                )
            );
    }
}
