/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.forums;

import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.observation.Event;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.ExpressionContext;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.forum.jcr.JCRThread;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;

/**
 * {@link ActivityType} implementation for thread-related activities
 */
public class ThreadActivityType extends AbstractWorkspacesActivityType
{

    /** data name for the thread id */
    public static final String THREAD_ID = "threadId";
    /** data name for the thread title */
    public static final String THREAD_TITLE = "threadTitle";
    /** data name for the thread author */
    public static final String THREAD_AUTHOR = "threadAuthor";
    /** Constant for activity's category */
    public static final String ACTIVITY_CATEGORY_THREADS = "threads";
    /** data name for the thread id */
    public static final String THREAD_CATEGORY = "threadCategory";
    /** data name for the thread creation date */
    public static final String THREAD_CREATIONDATE = "threadCreationDate";
    
    /** Thread Workspace Module */
    protected ForumWorkspaceModule _forumModule;
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _forumModule = moduleManagerEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
    }

    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        JCRThread thread = (JCRThread) parameters.get(ObservationConstants.ARGS_THREAD);
        activity.setValue(THREAD_TITLE, thread.getTitle());
        activity.setValue(THREAD_AUTHOR, thread.getAuthor());
        activity.setValue(THREAD_CATEGORY, thread.getCategory().name());
        activity.setValue(THREAD_ID, thread.getId());
        activity.setValue(PROJECT_CATEGORY, ACTIVITY_CATEGORY_THREADS);
        activity.setValue(THREAD_CREATIONDATE, thread.getCreationDate());
        activity.setValue(THREAD_CATEGORY, thread.getCategory().name());
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        String threadId = activity.getValue(THREAD_ID);
        
        String projectName = activity.getValue(PROJECT_NAME);
        Project project = _projectManager.getProject(projectName);
        if (!ObservationConstants.EVENT_THREAD_DELETED.equals(activity.getEventType()))
        {
            json.put("threadUrl", _forumModule.getThreadUri(project, threadId));
        }
        else
        {
            json.put("threadUrl", _forumModule.getModuleUrl(project));
        }

        return json;
    }
    
    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        if (!super.isMergeable(activity1, activity2))
        {
            return false;
        }
        
        String thread1 = activity1.getValue(THREAD_ID);
        String thread2 = activity1.getValue(THREAD_ID);
        
        return thread1 != null && thread2 != null && thread1.equals(thread2);
    }

    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        List<Map<String, Object>> mergedThreads = new ArrayList<>();
        
        for (Activity activity : activities)
        {
            Map<String, Object> json = activity.toJSONForClient();
            Map<String, Object> threadInfo = new HashMap<>();
            
            threadInfo.put(THREAD_TITLE, json.get(THREAD_TITLE));
            threadInfo.put(THREAD_ID, json.get(THREAD_ID));
            threadInfo.put("threadUrl", json.get("threadUrl"));
            
            mergedThreads.add(threadInfo);
        }
        mergedActivities.put("threads", mergedThreads);
        
        return mergedActivities;
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        return new StringExpression(THREAD_TITLE, Operator.WD, pattern, ExpressionContext.newInstance().withCaseInsensitive(true));
    }

    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        JCRThread thread = (JCRThread) args.get(ObservationConstants.ARGS_THREAD);
        
        return getParentProject(thread);
    }
    
    /**
     * Check if activity is generic type, and should be handled by ThreadsActivityNotifier, or a more specific notifier
     * @return true if activity is generic type
     */
    public boolean isGenericThreadActivityType()
    {
        return true;
    }
}
