/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.forums;

import java.util.Map;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;

/**
 * {@link ActivityType} implementation for thread-comments-related activities
 */
public class ThreadCommentActivityType extends ThreadActivityType
{
    /** data name for the post message */
    public static final String CONTENT = "content";
    /** data name for the post id*/
    public static final String COMMENT_ID = "commentId";
    /** data name for the comment author */
    public static final String COMMENT_AUTHOR = "commentAuthor";
    /** data name for the comment author name*/
    public static final String COMMENT_AUTHOR_NAME = "commentAuthorName";
    /** data name for the comment creation date */
    public static final String COMMENT_CREATIONDATE = "commentCreationDate";

    /** Thread Workspace Module */
    protected ThreadJSONHelper _threadJSONHelper;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _threadJSONHelper = (ThreadJSONHelper) serviceManager.lookup(ThreadJSONHelper.ROLE);
    }

    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);

        RichTextComment comment = (RichTextComment)  parameters.get(ObservationConstants.ARGS_THREAD_COMMENT);

        activity.setValue(COMMENT_ID, comment.getId());
        activity.setValue(COMMENT_AUTHOR, comment.getAuthor());
        activity.setValue(COMMENT_AUTHOR_NAME, comment.getAuthorName());
        activity.setValue(COMMENT_CREATIONDATE, comment.getCreationDate());
        String textContent = _threadJSONHelper.richTextToSimpleText(comment.getRichTextContent(), 150);
        activity.setValue(CONTENT, textContent);
    }

    @Override
    public boolean isMergeable(Activity activity1, Activity activity2)
    {
        return false;
    }
}
