/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.activities.projects;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;

import javax.jcr.RepositoryException;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.group.Group;
import org.ametys.core.group.GroupIdentity;
import org.ametys.core.group.GroupManager;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityType;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.JCRProjectMember;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;

/**
 * {@link ActivityType} implementation for the addition of a member
 */
public class MemberAddedActivityType extends AbstractProjectsActivityType
{
    /** data name for the member type */
    public static final String MEMBER_TYPE = "memberType";
    /** data name for the member identity */
    public static final String MEMBER = "member";
    
    private GroupManager _groupManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        super.service(serviceManager);
        _groupManager = (GroupManager) serviceManager.lookup(GroupManager.ROLE);
    }
    
    @Override
    public boolean supports(Event event)
    {
        return Config.getInstance().getValue("workspaces.member.added.send.notification", true, false) // Check feature enabled
                && _isNewUser(event);
    }
    
    private boolean _isNewUser(Event event)
    {
        Map<String, Object> args = event.getArguments();
        Project project = getProjectFromEvent(event);
        JCRProjectMember member = (JCRProjectMember) args.get(ObservationConstants.ARGS_MEMBER);
        if (MemberType.USER.equals(member.getType()))
        {
            if (_userIsPartOfProjectGroups(member.getUser(), project))
            {
                // if the user is already present via a group, we ignore the event
                return false;
            }
        }
        return true;
    }
    
    private boolean _userIsPartOfProjectGroups(UserIdentity identity, Project project)
    {
        boolean alreadyPresent = _projectMemberManager.getProjectMembers(project, false)
                .stream()
                // retrieve the group
                .map(ProjectMember::getGroup)
                .filter(Objects::nonNull)
                // retrieve group users
                .map(Group::getUsers)
                // check if a group contains our user
                .anyMatch(users -> users.contains(identity));
        return alreadyPresent;
    }
    
    @Override
    public void setAdditionalActivityData(Activity activity, Map<String, Object> parameters) throws RepositoryException
    {
        super.setAdditionalActivityData(activity, parameters);
        
        JCRProjectMember member = (JCRProjectMember) parameters.get(org.ametys.plugins.workspaces.ObservationConstants.ARGS_MEMBER);
        member = _resolver.resolveById(member.getId());
        activity.setValue(MEMBER, MemberType.GROUP == member.getType()
                ? GroupIdentity.groupIdentityToString(member.getGroup())
                : UserIdentity.userIdentityToString(member.getUser()));
        activity.setValue(MEMBER_TYPE, member.getType().name().toLowerCase());
    }
    
    @Override
    public Map<String, Object> additionnalDataToJSONForClient(Activity activity)
    {
        // Override the MEMBER element to provide the fullName or Label instead of the identity
        Map<String, Object> json = super.additionnalDataToJSONForClient(activity);
        
        String memberIdentity = activity.getValue(MEMBER);
        String type = activity.hasValue(MEMBER_TYPE) ? activity.getValue(MEMBER_TYPE) : MemberType.USER.name();
        if (MemberType.GROUP == MemberType.valueOf(type.toUpperCase()))
        {
            GroupIdentity groupIdentity = GroupIdentity.stringToGroupIdentity(memberIdentity);
            Group group = _groupManager.getGroup(groupIdentity);
            json.put(MEMBER, group != null ? group.getLabel() : memberIdentity);
        }
        else if (MemberType.USER == MemberType.valueOf(type.toUpperCase()))
        {
            UserIdentity userIdentity = UserIdentity.stringToUserIdentity(memberIdentity);
            User user = _userManager.getUser(userIdentity);
            
            json.put(MEMBER, user != null ? user.getFullName() : (userIdentity != null ? userIdentity.getLogin() : memberIdentity));
        }
        json.put("identity", memberIdentity);
        
        return json;
    }
    
    @Override
    public Expression getFilterPatternExpression(String pattern)
    {
        return null;
    }
    
    @Override
    public Map<String, Object> mergeActivities(List<Activity> activities)
    {
        Map<String, Object> mergedActivities = super.mergeActivities(activities);
        
        Map<String, Map<String, Object>> members = new HashMap<>();
        
        for (Activity activity : activities)
        {
            if (activity.hasValue(MEMBER))
            {
                Map<String, Object> json = activity.toJSONForClient();
                String identity = (String) json.get("identity");
                if (!members.containsKey(identity))
                {
                    Map<String, Object> m = new HashMap<>();
                    m.put("name", json.get(MEMBER));
                    m.put("type", json.get(MEMBER_TYPE));
                    m.put("identity", identity);
                    members.put(identity, m);
                }
            }
        }
        
        mergedActivities.put("members", members.values());
        return mergedActivities;
    }
    
    @Override
    public Project getProjectFromEvent(Event event)
    {
        Map<String, Object> args = event.getArguments();
        
        return (Project) args.get(ObservationConstants.ARGS_PROJECT);
    }
}
