/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.archive;

import java.io.IOException;
import java.io.InputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.Files;
import java.nio.file.Path;
import java.util.Optional;
import java.util.zip.ZipEntry;
import java.util.zip.ZipOutputStream;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.sax.TransformerHandler;
import javax.xml.transform.stream.StreamResult;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

import org.ametys.plugins.contentio.archive.Archivers;
import org.ametys.plugins.contentio.archive.Archivers.AmetysObjectNotImportedException;
import org.ametys.plugins.contentio.archive.ImportGlobalFailException;
import org.ametys.plugins.contentio.archive.ImportReport;
import org.ametys.plugins.contentio.archive.Merger;
import org.ametys.plugins.contentio.archive.ResourcesArchiverHelper;
import org.ametys.plugins.contentio.archive.SystemViewHandler;
import org.ametys.plugins.contentio.archive.UnitaryImporter;
import org.ametys.plugins.contentio.archive.ZipEntryHelper;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollectionFactory;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for operations related to exporting or import project.
 */
public class ProjectArchiverHelper extends AbstractLogEnabled implements Component, Serviceable
{
    /** the Avalon role */
    public static final String ROLE = ProjectArchiverHelper.class.getName();
    
    private static final String __RESOURCES_NODE_NAME = "ametys-internal:resources";
    private static final String __PROJECTS_NODE_NAME = "projects";
    
    private static final String __MODULE_ARCHIVE_FILENAME = "module.xml";
    private static final String __PROJECT_ARCHIVE_FILE = "project.xml";
    
    /** the ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** the resource archiver helper */
    protected ResourcesArchiverHelper _resourcesArchiverHelper;

    public void service(ServiceManager manager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _resourcesArchiverHelper = (ResourcesArchiverHelper) manager.lookup(ResourcesArchiverHelper.ROLE);
    }
    
    /**
     * Export the project to the zip archive
     * @param project the project to export
     * @param zos the zip output stream
     * @param prefix the prefix use for the path of all zip entry of the project
     * @throws IOException if an error occurred
     */
    public void exportProject(Project project, ZipOutputStream zos, String prefix) throws IOException
    {
        try
        {
            // Export the project data except the module resources
            ZipEntry projectEntry = new ZipEntry(prefix + __PROJECT_ARCHIVE_FILE);
            zos.putNextEntry(projectEntry);
            
            TransformerHandler handler = Archivers.newTransformerHandler();
            handler.setResult(new StreamResult(zos));
            
            Node projectNode = project.getNode();
            projectNode.getSession().exportSystemView(
                projectNode.getPath(),
                new SystemViewHandler(handler, name -> __RESOURCES_NODE_NAME.equals(name), __ -> false),
                false,
                false
            );
            
            // then the module resources if any, one by one
            if (projectNode.hasNode(__RESOURCES_NODE_NAME))
            {
                Node resources = projectNode.getNode(__RESOURCES_NODE_NAME);
                String modulePrefix = prefix + resources.getName() + "/";
                
                NodeIterator moduleIterator = resources.getNodes();
                while (moduleIterator.hasNext())
                {
                    Node moduleNode = moduleIterator.nextNode();
                    String name = moduleNode.getName();
                    // export documents as a file tree
                    if ("documents".equals(name))
                    {
                        ResourceCollection documentRoot = _resolver.resolve(moduleNode, false);
                        _resourcesArchiverHelper.exportCollection(documentRoot, zos, modulePrefix  + name + "/");
                    }
                    // serialize other module to an XML
                    else
                    {
                        ZipEntry moduleEntry = new ZipEntry(modulePrefix + name + "/" + __MODULE_ARCHIVE_FILENAME);
                        zos.putNextEntry(moduleEntry);
                        
                        moduleNode.getSession().exportSystemView(
                            moduleNode.getPath(),
                            new SystemViewHandler(handler, __ -> false, __ -> false),
                            false,
                            false
                        );
                    }
                }
                
            }
        }
        catch (Exception e)
        {
            throw new RuntimeException("Unable to archive project " + project.getName(), e);
        }
    }

    /**
     * Import archived projects data.
     * @param pluginNode the plugin node where the new projects node will be added
     * @param zipPath the path to the ZIP containing the data
     * @param baseImportProjectsPath the path in the ZIP where the projects data are located
     * @param merger the merger
     * @return the import report
     * @throws IOException if an error occurs
     */
    public ImportReport importProjects(Node pluginNode, Path zipPath, String baseImportProjectsPath, Merger merger) throws IOException
    {
        ImportReport report = new ImportReport();
        
        if (ZipEntryHelper.zipEntryFolderExists(zipPath, baseImportProjectsPath))
        {
            try
            {
                // projects data are included in the archive
                // start by getting the projects node in the repository
                Node projectsNode = pluginNode.hasNode(__PROJECTS_NODE_NAME)
                    ? pluginNode.getNode(__PROJECTS_NODE_NAME)
                    : pluginNode.addNode(__PROJECTS_NODE_NAME, RepositoryConstants.NAMESPACE_PREFIX + ":unstructured");
                
                // use an importer to delegate the handling of merge
                ProjectImporter projectImporter = new ProjectImporter(zipPath, projectsNode, merger, report);
                
                try (DirectoryStream<Path> projectPaths = ZipEntryHelper.children(
                        zipPath,
                        Optional.of(baseImportProjectsPath.toString()),
                        Files::isDirectory))
                {
                    for (Path projectPath : projectPaths)
                    {
                        projectImporter.unitaryImport(zipPath, projectPath, merger, getLogger());
                    }
                }
            }
            catch (ParserConfigurationException e)
            {
                throw new IOException(e);
            }
            catch (RepositoryException e)
            {
                throw new IOException(e);
            }
        }
        
        return report;
    }
    
    private Project _createProject(Path projectZipPath, Path zipPath, Node projectsNode, Merger merger) throws IOException, AmetysObjectNotImportedException
    {
        Path projectData = projectZipPath.resolve(__PROJECT_ARCHIVE_FILE);
        try
        {
            Session session = projectsNode.getSession();
            String parentAbsPath = projectsNode.getPath();
            getLogger().info("XML from '{}!{}' will be imported to '{}' with implementation of merger '{}'", zipPath, projectData.toString(), parentAbsPath, merger);
            try (InputStream in = ZipEntryHelper.zipEntryFileInputStream(zipPath, projectData.toString()))
            {
                merger.jcrImportXml(session, parentAbsPath, in);
            }
            
            Archivers.unitarySave(projectsNode, getLogger());
            
            Node projectNode = projectsNode.getNode(projectZipPath.getFileName().toString());
            
            _importModules(projectZipPath, zipPath, merger, session, projectNode);
            
            return _resolver.resolve(projectNode, false);
        }
        catch (RepositoryException e)
        {
            throw new IOException(e);
        }

    }

    private void _importModules(Path projectZipPath, Path zipPath, Merger merger, Session session, Node projectNode) throws IOException, RepositoryException
    {
        Path modulesEntryPath = projectZipPath.resolve(__RESOURCES_NODE_NAME);
        if (ZipEntryHelper.zipEntryFolderExists(zipPath, modulesEntryPath.toString()))
        {
            Node modulesRoot = projectNode.hasNode(__RESOURCES_NODE_NAME)
                    ? projectNode.getNode(__RESOURCES_NODE_NAME)
                            : projectNode.addNode(__RESOURCES_NODE_NAME, JCRResourcesCollectionFactory.RESOURCESCOLLECTION_NODETYPE);
            
            try (DirectoryStream<Path> moduleEntries = ZipEntryHelper.children(zipPath, Optional.of(modulesEntryPath.toString()), Files::isDirectory))
            {
                for (Path moduleEntryPath : moduleEntries)
                {
                    // File module is stored as a collection of file
                    if ("documents".equals(moduleEntryPath.getFileName().toString()))
                    {
                        _resourcesArchiverHelper.importCollection(moduleEntryPath.toString() + "/", modulesRoot, zipPath, merger);
                    }
                    // other modules are serialized as XML
                    else
                    {
                        try (InputStream moduleIn = ZipEntryHelper.zipEntryFileInputStream(zipPath, moduleEntryPath.resolve(__MODULE_ARCHIVE_FILENAME).toString()))
                        {
                            getLogger().info("XML from '{}!{}' will be imported to '{}' with implementation of merger '{}'", zipPath, moduleEntryPath.toString(), modulesRoot.getPath().toString(), merger);
                            merger.jcrImportXml(session, modulesRoot.getPath(), moduleIn);
                            session.save();
                        }
                    }
                }
            }
        }
    }
    
    /**
     * Inner class to leverage the logic from UnitaryImporter
     */
    protected class ProjectImporter implements UnitaryImporter<Project>
    {
        private DocumentBuilder _builder;
        private Path _zipPath;
        private Merger _merger;
        private Node _projectsNode;
        private ImportReport _report;

        /**
         * Build an new importer
         * @param zipPath the path to the archive ZIP
         * @param projectsNode the node where the data will be imported
         * @param merger the merger
         * @param report the report
         * @throws ParserConfigurationException if an error occurred
         */
        public ProjectImporter(Path zipPath, Node projectsNode, Merger merger, ImportReport report) throws ParserConfigurationException
        {
            DocumentBuilderFactory builderFactory = DocumentBuilderFactory.newInstance();
            builderFactory.setNamespaceAware(true);
            _builder = builderFactory
                    .newDocumentBuilder();
            _merger = merger;
            _projectsNode = projectsNode;
            _zipPath = zipPath;
            _report = report;
        }

        public String objectNameForLogs()
        {
            return "Project";
        }

        public Document getPropertiesXml(Path projectPath) throws Exception
        {
            String propertiesPath = projectPath.resolve(__PROJECT_ARCHIVE_FILE).toString();
            try (InputStream stream = ZipEntryHelper.zipEntryFileInputStream(_zipPath, propertiesPath))
            {
                Document doc = _builder.parse(stream);
                return doc;
            }
            catch (SAXException e)
            {
                throw new IOException(e);
            }
        }

        public String retrieveId(Document propertiesXml) throws Exception
        {
            return "project://" + Archivers.xpathEvalNonEmpty("sv:node/sv:property[@sv:name='jcr:uuid']/sv:value", propertiesXml);
        }

        public Project create(Path projectZipPath, String id, Document propertiesXml) throws AmetysObjectNotImportedException, ImportGlobalFailException, Exception
        {
            return _createProject(projectZipPath, _zipPath, _projectsNode, _merger);
        }

        public ImportReport getReport()
        {
            return _report;
        }
        
    }
}
