/*
 *  Copyright 2014 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.calendars.events;

import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.util.List;
import java.util.Optional;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.messagingconnector.EventRecurrenceTypeEnum;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.tag.TaggableAmetysObject;
import org.ametys.plugins.workspaces.calendars.Calendar;

/**
 * Calendar event interface
 */
public interface CalendarEvent extends TaggableAmetysObject
{
    /**
     * Get the title of the event
     * @return the title of the event
     */
    public String getTitle();
    
    /**
     * Get the calendar holding the event
     * @return the calendar
     */
    public Calendar getCalendar();
    
    /**
     * Get the description of the event
     * @return the description of the event
     */
    public String getDescription();
    
    /**
     * Get the location of the event
     * @return the location of the event
     */
    public String getLocation();
    
    /**
     * Get the date of the begin of the event
     * @return the date of the event
     */
    public ZonedDateTime getStartDate();

    /**
     * Get the date of the end of the event
     * @return the date of the event
     */
    public ZonedDateTime getEndDate();

    /**
     * Get the date zone of the event
     * @return the date zone of the event
     */
    public ZoneId getZone();
    
    /**
     * Get if the event last all the day
     * @return true if the event last all the day
     */
    public Boolean getFullDay();
    
    /**
     * Retrieves the the creator.
     * @return the creator.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getCreator();
    
    /**
     * Retrieves the creation date.
     * @return the creation date.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ZonedDateTime getCreationDate();
    
    /**
     * Retrieves the last contributor.
     * @return the last contributor.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getLastContributor();
    
    /**
     * Retrieves the last modification date.
     * @return the last modification date.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public ZonedDateTime getLastModified();
     
    /**
     * Retrieves the recurrence type.
     * @return the recurrence type.
     */
    public EventRecurrenceTypeEnum getRecurrenceType();
    
    /**
     * Retrieves if the event is recurrent.
     * @return true if the event is recurrent.
     */
    public Boolean isRecurrent();
    
    /**
     * Retrieves the end date of the frequency.
     * @return the end date of the frequency.
     */
    public ZonedDateTime getRepeatUntil();
    
    /**
     * Retrieves the list of excluded event date.
     * @return the list of excluded event date.
     */
    public List<ZonedDateTime> getExcludedOccurences();
    
    /**
     * Retrieves the list of all event date between startDate and endDate.
     * @param startDate the start date
     * @param endDate the end date
     * @return the list of all event date between startDate and endDate.
     */
    public List<CalendarEventOccurrence> getOccurrences(ZonedDateTime startDate, ZonedDateTime endDate);
    
    /**
     * Retrieves the start date of the first Event which end after the date
     * @param date the date
     * @return the start date of the first Event which end after the date
     */
    public Optional<CalendarEventOccurrence> getFirstOccurrence(ZonedDateTime date);
    
    /**
     * Retrieves the date of the next event after the date
     * @param occurrence the current occurrence
     * @return the occurrence of the next event after the date
     */
    public Optional<CalendarEventOccurrence> getNextOccurrence(CalendarEventOccurrence occurrence);
    
    /**
     * Retrieves the organiser.
     * @return the organiser.
     * @throws AmetysRepositoryException if an error occurs.
     */
    public UserIdentity getOrganiser();

    /**
     * Retrieves the resource list.
     * @return the resource list.
     */
    public List<String> getResources();
    
}
