/*
 *  Copyright 2024 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.comments;

import java.util.List;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;

import org.ametys.cms.repository.comment.RichTextComment;
import org.ametys.core.observation.Event;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.forum.ForumWorkspaceModule;
import org.ametys.plugins.workspaces.forum.Thread;
import org.ametys.plugins.workspaces.forum.json.ThreadJSONHelper;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.WebHelper;

/**
 * Observer to send mails to mentioned users in contributor comments of {@link Thread}
 */
public class NotifyThreadCommentMentionsObserver extends AbstractNotifyWorkspacesCommentMentionsObserver<Thread>
{
    
    /** The forum module */
    protected ForumWorkspaceModule _forumModule;
    
    /** The project tags DAO */
    protected ThreadJSONHelper _threadJSONHelper;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _forumModule = moduleManagerEP.getModule(ForumWorkspaceModule.FORUM_MODULE_ID);
        _threadJSONHelper = (ThreadJSONHelper) manager.lookup(ThreadJSONHelper.ROLE);
    }

    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public boolean supports(Event event)
    {
        return event.getId().equals(ObservationConstants.EVENT_THREAD_COMMENTED);
    }
    
    @Override
    protected String _getAmetysObjectTitle(Thread thread)
    {
        return thread.getTitle();
    }

    @Override
    protected I18nizableText _getMailMessage(MentionableObject mentionableObject)
    {
        Thread thread = (Thread) mentionableObject.ametysObject();
        Project project = _projectManager.getParentProject(thread);
        User author = mentionableObject.author();
        List<String> i18nParams = List.of(author.getFullName(), _getAmetysObjectTitle(thread), project.getTitle());
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_THREAD_COMMENT_MENTION_MAIL_MESSAGE", i18nParams);
    }
    
    private String _getContent(RichTextComment comment, Project project) throws Exception
    {
        Request request = ContextHelper.getRequest(_context);
        String siteName = WebHelper.getSiteName(request);
        if (siteName == null)
        {
            request.setAttribute("site", project.getSite().getName());
        }
        return _threadJSONHelper.richTextToRendering(comment.getRichTextContent());
    }
    
    @Override
    protected String _transformSyntaxTextToReadableTextWithColors(String content, UserIdentity recipient)
    {
        return _mentionUtils.transformRichTextToReadableText(content, recipient);
    }

    @Override
    protected MentionableObject _getMentionableObjectFromArguments(Map<String, Object> arguments) throws Exception
    {
        Thread thread = _getAmetysObjectFromArguments(arguments);
        RichTextComment comment = (RichTextComment) arguments.get(ObservationConstants.ARGS_THREAD_COMMENT);
        comment = thread.getComment(comment.getId());
        Project project = _projectManager.getParentProject(thread);
        String url = _getAbsoluteUrl(thread, project);
        String content = _getContent(comment, project);

        return new MentionableObject(
            _userManager.getUser(comment.getAuthor()),
            content,
            _mentionUtils.extractMentionedUsersFromRichText(content),
            comment.getCreationDate(),
            thread,
            new LinkToAmetysObject(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_FORUMTHREAD_BUTTON_TEXT")),
            _getLanguage(project)
        );
    }

    @Override
    protected I18nizableText _getMailMessageType()
    {
        return new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_THREAD_MENTION_MAIL_THREAD_COMMENT_TITLE");
    }
    
    @Override
    protected String _getModuleId()
    {
        return ForumWorkspaceModule.FORUM_MODULE_ID;
    }

    @Override
    protected String getUrl(Thread thread, Project project)
    {
        return _forumModule.getThreadUri(project, thread.getId());
    }
}
