/*
 *  Copyright 2017 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.dav;

import java.io.IOException;
import java.io.InputStream;
import java.time.ZoneId;
import java.time.format.DateTimeFormatter;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.TimeZone;

import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.environment.Response;
import org.apache.cocoon.environment.http.HttpEnvironment;
import org.apache.cocoon.generation.AbstractGenerator;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.tuple.Pair;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObject;

/**
 * Reader for WebDAV PROFIND method
 */
public class WebdavPropfindGenerator extends AbstractGenerator
{
    /** The webdav namespace */
    public static final String WEBDAV_NAMESPACE = "DAV:";

    /** Default recursion depth for folders */
    public static final int DEFAULT_DEPTH_ALLPROP = 1;
    
    /** Default recursion depth for no recursion */
    public static final int DEFAULT_DEPTH_NO_RECURSION = 0;
    
    private static final List<Pair<String, String>> __DEFAULT_PROPS_RESOURCE = Arrays.asList(Pair.of("DAV:", "creationdate"), 
                                                                                             Pair.of("DAV:", "getlastmodified"),
                                                                                             Pair.of("DAV:", "getcontentlength"), 
                                                                                             Pair.of("DAV:", "displayname"),
                                                                                             Pair.of("DAV:", "contenttype"),
                                                                                             Pair.of("DAV:", "resourcetype"));

    private static final List<Pair<String, String>> __DEFAULT_PROPS_COLLECTION = Arrays.asList(Pair.of("DAV:", "displayname"),
                                                                                               Pair.of("DAV:", "resourcetype"));
    
    private enum PropfindType
    {
        PROP,
        ALLPROP,
        PROPNAME
    }

    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        AmetysObject resource = (AmetysObject) request.getAttribute("resource");
        
        HttpServletRequest httpRequest = (HttpServletRequest) objectModel.get(HttpEnvironment.HTTP_REQUEST_OBJECT);
        
        Document document = null;
        try (InputStream is = httpRequest.getInputStream())
        {
            DocumentBuilderFactory documentBuilderFactory = DocumentBuilderFactory.newInstance();
            documentBuilderFactory.setNamespaceAware(true);
            document = documentBuilderFactory.newDocumentBuilder().parse(is);
        }
        catch (ParserConfigurationException e)
        {
            throw new ProcessingException(e);
        }
        
        Element propfindRoot = document.getDocumentElement();
        Element propfindRequest = _getFirstChildElement(propfindRoot);
        
        String requestName = propfindRequest.getLocalName();
        PropfindType type = PropfindType.valueOf(requestName.toUpperCase());
        
        List<Pair<String, String>> props = null;
        if (type == PropfindType.PROP)
        {
            props = _getProps(propfindRequest);
        }

        Response response = ObjectModelHelper.getResponse(objectModel);
        response.setHeader("Content-Type", "application/xml; charset=utf-8");
        
        HttpServletResponse httpResponse = (HttpServletResponse) objectModel.get(HttpEnvironment.HTTP_RESPONSE_OBJECT);
        httpResponse.setStatus(207);

        contentHandler.startDocument();
        contentHandler.startPrefixMapping("d", WEBDAV_NAMESPACE);
        XMLUtils.startElement(contentHandler, "d:multistatus");
        
        if (resource instanceof Resource)
        {
            _processResource((Resource) resource, type, props);
        }
        else if (resource instanceof ResourceCollection)
        {
            // If we request ALLPROP, by default we will require 1 recursion, else no recursion (by default, so this is overridable by Depth)
            int maxDepth = type == PropfindType.ALLPROP ? DEFAULT_DEPTH_ALLPROP : DEFAULT_DEPTH_NO_RECURSION;
            try
            {
                String depthHeader = httpRequest.getHeader("Depth");
                if (!StringUtils.isEmpty(depthHeader))
                {
                    maxDepth = Integer.parseInt(depthHeader);
                }
            }
            catch (NumberFormatException e)
            {
                // Nothing, depth will be the default value
            }
                
            _processCollection((ResourceCollection) resource, 0, maxDepth, type, props);
        }
        
        XMLUtils.endElement(contentHandler, "d:multistatus");
        contentHandler.endDocument();
    }
    
    private Element _getFirstChildElement(Element element)
    {
        NodeList childNodes = element.getChildNodes();
        
        for (int i = 0; i < childNodes.getLength(); i++)
        {
            Node node = childNodes.item(i);
            
            if (node instanceof Element)
            {
                return (Element) node;
            }
        }
        
        throw new IllegalArgumentException("Unrecognized propfind request");
    }
    
    private void _processResource(Resource resource, PropfindType type, List<Pair<String, String>> props) throws SAXException
    {
        String href = ResolveURIComponent.resolve("webdav-project-resource", resource.getId(), false, true);
        
        _startResponseNode(href);
        _startPropstatNode();
        
        List<Pair<String, String>> actualProps = props;
        
        if (type == PropfindType.PROPNAME)
        {
            XMLUtils.createElement(contentHandler, "d:creationdate");
            XMLUtils.createElement(contentHandler, "d:getlastmodified");
            XMLUtils.createElement(contentHandler, "d:getcontentlength");
            XMLUtils.createElement(contentHandler, "d:getcontenttype");
            XMLUtils.createElement(contentHandler, "d:resourcetype");
            XMLUtils.createElement(contentHandler, "d:supportedlock");
            
            _endPropstatNode("HTTP/1.1 200 OK");
            _endResponseNode();
            return;
        }
        
        if (type == PropfindType.ALLPROP)
        {
            actualProps = __DEFAULT_PROPS_RESOURCE;
        }
        
        List<Pair<String, String>> notFound = new ArrayList<>();
        
        // get all asked properties
        for (Pair<String, String> prop : actualProps)
        {
            String namespace = prop.getLeft();
            String name = prop.getRight();
            boolean found = _processProperty(namespace, name, resource, href);
            
            if (!found)
            {
                //add to not found
                notFound.add(prop);
            }
        }
        
        _endPropstatNode("HTTP/1.1 200 OK");
        _notFoundPropstatNode(notFound);
        _endResponseNode();
    }
    
    private boolean _processProperty(String namespace, String name, Resource resource, String href) throws SAXException
    {
        boolean found = false;
        
        if (namespace.equals(WEBDAV_NAMESPACE))
        {
            if (name.equals("creationdate"))
            {
                String lastModified = DateTimeFormatter.ISO_INSTANT.format(DateUtils.asInstant(resource.getLastModified()));
                XMLUtils.createElement(contentHandler, "d:creationdate", lastModified);
                found = true;
            }
            else if (name.equals("getlastmodified"))
            {
                String lastModified = DateTimeFormatter.RFC_1123_DATE_TIME.withZone(TimeZone.getTimeZone("GMT").toZoneId()).format(DateUtils.asZonedDateTime(resource.getLastModified(), ZoneId.systemDefault()));
                XMLUtils.createElement(contentHandler, "d:getlastmodified", lastModified);
                found = true;
            }
            else if (name.equals("getcontentlength"))
            {
                XMLUtils.createElement(contentHandler, "d:getcontentlength", String.valueOf(resource.getLength()));
                found = true;
            }
            else if (name.equals("displayname"))
            {
                XMLUtils.createElement(contentHandler, "d:displayname", resource.getName());
                found = true;
            }
            else if (name.equals("contenttype"))
            {
                XMLUtils.createElement(contentHandler, "d:contenttype", resource.getMimeType());
                found = true;
            }
            else if (name.equals("getcontenttype"))
            {
                XMLUtils.createElement(contentHandler, "d:getcontenttype", resource.getMimeType());
                found = true;
            }
            else if (name.equals("resourcetype"))
            {
                XMLUtils.createElement(contentHandler, "d:resourcetype");
                found = true;
            }
            else if (name.equals("supportedlock"))
            {
                XMLUtils.startElement(contentHandler, "d:supportedlock");
                XMLUtils.startElement(contentHandler, "d:lockentry");
                XMLUtils.startElement(contentHandler, "d:lockscope");
                XMLUtils.createElement(contentHandler, "d:exclusive");
                XMLUtils.endElement(contentHandler, "d:lockscope");
                XMLUtils.startElement(contentHandler, "d:locktype");
                XMLUtils.createElement(contentHandler, "d:write");
                XMLUtils.endElement(contentHandler, "d:locktype");
                XMLUtils.endElement(contentHandler, "d:lockentry");
                XMLUtils.endElement(contentHandler, "d:supportedlock");
                found = true;
            }
        }
        else if (namespace.equals("http://ucb.openoffice.org/dav/props/"))
        {
            if (name.equals("IsReadOnly"))
            {
                contentHandler.startPrefixMapping("o", "http://ucb.openoffice.org/dav/props/");
                XMLUtils.createElement(contentHandler, "o:IsReadOnly", "false");
                contentHandler.endPrefixMapping("o");
                found = true;
            }
            else if (name.equals("BaseURI"))
            {
                contentHandler.startPrefixMapping("o", "http://ucb.openoffice.org/dav/props/");
                XMLUtils.createElement(contentHandler, "o:BaseURI", href);
                contentHandler.endPrefixMapping("o");
                found = true;
            }
            else if (name.equals("ObjectId"))
            {
                contentHandler.startPrefixMapping("o", "http://ucb.openoffice.org/dav/props/");
                XMLUtils.createElement(contentHandler, "o:ObjectId", resource.getId());
                contentHandler.endPrefixMapping("o");
                found = true;
            }
        }
        
        return found;
    }

    /**
     * Add a folder in the XML response
     * @param resource the resource to sax
     * @param props the list of properties to return
     * @param currentDepth current depth
     * @param maxDepth max depth (if current depth &gt;= max depth, no children nodes will be added, only the current folder)
     * @param type the prop find type
     * @throws SAXException an exception occurred
     */
    private void _processCollection(ResourceCollection resource, int currentDepth, int maxDepth, PropfindType type, List<Pair<String, String>> props) throws SAXException
    {
        String href = ResolveURIComponent.resolve("webdav-project-resource", resource.getId(), false, true) + "/";

        _startResponseNode(href);
        _startPropstatNode();
        
        List<Pair<String, String>> actualProps = props;
        
        if (type == PropfindType.PROPNAME)
        {
            XMLUtils.createElement(contentHandler, "d:displayname");
            XMLUtils.createElement(contentHandler, "d:resourcetype");
            
            _endPropstatNode("HTTP/1.1 200 OK");
            _endResponseNode();
            return;
        }
        
        if (type == PropfindType.ALLPROP)
        {
            actualProps = __DEFAULT_PROPS_COLLECTION;
        }
        
        List<Pair<String, String>> notFound = new ArrayList<>();

        // get all asked properties
        for (Pair<String, String> prop : actualProps)
        {
            String namespace = prop.getLeft();
            String name = prop.getRight();
            boolean found = false;
            
            if (namespace.equals(WEBDAV_NAMESPACE))
            {
                if (name.equals("displayname"))
                {
                    XMLUtils.createElement(contentHandler, "d:displayname", resource.getName());
                    found = true;
                }
                else if (name.equals("resourcetype"))
                {
                    XMLUtils.startElement(contentHandler, "d:resourcetype");
                    XMLUtils.createElement(contentHandler, "d:collection");
                    XMLUtils.endElement(contentHandler, "d:resourcetype");
                    found = true;
                }
            }
            
            if (!found)
            {
                //add to not found
                notFound.add(prop);
            }
        }
        
        _endPropstatNode("HTTP/1.1 200 OK");
        _notFoundPropstatNode(notFound);
        _endResponseNode();

        if (currentDepth < maxDepth)
        {
            for (AmetysObject child : resource.getChildren())
            {
                if (child instanceof Resource)
                {
                    _processResource((Resource) child, type, props);
                }
                else if (child instanceof ResourceCollection)
                {
                    _processCollection((ResourceCollection) child, currentDepth + 1, maxDepth, type, props);
                }
            }
        }
    }

    private void _startResponseNode(String href) throws SAXException
    {
        XMLUtils.startElement(contentHandler, "d:response");
        XMLUtils.createElement(contentHandler, "d:href", href);
    }

    private void _startPropstatNode()  throws SAXException
    {
        XMLUtils.startElement(contentHandler, "d:propstat");
        XMLUtils.startElement(contentHandler, "d:prop");
    }

    private void _endResponseNode() throws SAXException
    {
        XMLUtils.endElement(contentHandler, "d:response");
    }

    private void _endPropstatNode(String status) throws SAXException
    {
        XMLUtils.endElement(contentHandler, "d:prop");
        XMLUtils.createElement(contentHandler, "d:status", status);
        XMLUtils.endElement(contentHandler, "d:propstat");
    }

    private void _notFoundPropstatNode(List<Pair<String, String>> notFound) throws SAXException
    {
        if (notFound != null && !notFound.isEmpty())
        {
            _startPropstatNode();
            
            for (Pair<String, String> prop : notFound)
            {
                String namespace = prop.getLeft();
                String name = prop.getRight();
                
                contentHandler.startPrefixMapping("", namespace);
                XMLUtils.createElement(contentHandler, name);
                contentHandler.endPrefixMapping("");
            }
            
            _endPropstatNode("HTTP/1.1 404 Not Found");
        }
    }
    
    private List<Pair<String, String>> _getProps(Element props)
    {
        List<Pair<String, String>> result = new ArrayList<>();
        NodeList propNames = props.getChildNodes();
        for (int j = 0; j < propNames.getLength(); j++)
        {
            Node propName = propNames.item(j);
            
            if (propName.getNodeType() != Node.ELEMENT_NODE)
            {
                continue;
            }
            
            result.add(Pair.of(propName.getNamespaceURI(), propName.getLocalName()));
        }

        return result;
    }
}
