/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents;

import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.observation.AsyncObserver;
import org.ametys.core.observation.Event;
import org.ametys.core.ui.mail.StandardMailBodyHelper;
import org.ametys.core.ui.mail.StandardMailBodyHelper.MailBodyBuilder;
import org.ametys.plugins.explorer.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.runtime.i18n.I18nizableTextParameter;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;

import jakarta.mail.MessagingException;

/**
 * Observer to send mails to manager if the workspace space is almost full or full
 */
public class SendMailOnFileManagerStorageSpaceLimitreachedObserver implements AsyncObserver, Serviceable, Contextualizable
{
    /** The documents module DAO */
    protected WorkspaceExplorerResourceDAO _workspaceExplorerResourceDAO;
    
    /** Project manager */
    protected ProjectManager _projectManager;
    
    /** The workspaces helper */
    protected WorkspacesHelper _workspaceHelper;
    
    /** The document module */
    protected DocumentWorkspaceModule _documentModule;
    
    /** The context */
    protected Context _context;

    /** The rendering context handler */
    protected RenderingContextHandler _renderingContextHandler;

    public void service(ServiceManager manager) throws ServiceException
    {
        _workspaceExplorerResourceDAO = (WorkspaceExplorerResourceDAO) manager.lookup(WorkspaceExplorerResourceDAO.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);

        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) manager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _documentModule = moduleManagerEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
    }
    
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    public boolean supports(Event event)
    {
        return ObservationConstants.EVENT_RESOURCE_CREATED.equals(event.getId());
    }

    public int getPriority()
    {
        return Integer.MAX_VALUE;
    }

    public void observe(Event event, Map<String, Object> transientVars) throws Exception
    {
        String parenIt = (String) event.getArguments().get(ObservationConstants.ARGS_PARENT_ID);

        Long storageSpaceLimit = _workspaceExplorerResourceDAO.getStorageSpaceLimit();
        Project project = _projectManager.getParentProject(parenIt);
        Long usedStorageSpace = _workspaceExplorerResourceDAO.getUsedStorageSpaceByProject(project);

        if (storageSpaceLimit != null && usedStorageSpace >= storageSpaceLimit)
        {
            _sendMail(project, true); // Send mail for 100% reached
            return; // return, as we do not want to send the 95% mail as well
        }

        if (storageSpaceLimit != null && usedStorageSpace >= (storageSpaceLimit * 0.95))
        {
            _sendMail(project, false); // Send mail for 95% reached
        }
        
    }

    private void _sendMail(Project project, boolean isFull) throws MessagingException, IOException
    {
        String part = isFull ? "FULL" : "ALMOST_FULL";

        String mailFrom = Config.getInstance().getValue("smtp.mail.from");
        
        Map<String, I18nizableTextParameter> params = new HashMap<>();
        params.put("project", new I18nizableText(project.getTitle()));
        params.put("projectTitle", new I18nizableText(project.getTitle()));
        params.put("projectUrl", new I18nizableText(_projectManager.getProjectUrl(project, StringUtils.EMPTY)));
        
        String url = _getModuleURL(project);
        
        MailBodyBuilder htmlBodyBuilder = StandardMailBodyHelper.newHTMLBody()
            .withTitle(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STORAGE_SPACE_" + part + "_MANAGER_MAIL_TITLE", params))
            .withMessage(new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STORAGE_SPACE_" + part + "_MANAGER_MAIL_BODY_HTML", params))
            .withLink(url, new I18nizableText("plugin.workspaces", "PROJECT_MAIL_NOTIFICATION_BODY_DEFAULT_BUTTON_TEXT"));
        
        I18nizableText i18nSubject = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STORAGE_SPACE_" + part + "_MANAGER_MAIL_SUBJECT", params);
        I18nizableText i18nTextBody = new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_STORAGE_SPACE_" + part + "_MANAGER_MAIL_BODY_TEXT", params);
        _workspaceHelper.sendMailToManagers(project, mailFrom, i18nSubject, htmlBodyBuilder, i18nTextBody);
    }

    private String _getModuleURL(Project project)
    {
        Request request = ContextHelper.getRequest(_context);
        request.setAttribute("forceAbsoluteUrl", true);
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);
        String url = null;
        try
        {
            url = _documentModule.getModuleUrl(project);
        }
        finally
        {
            _renderingContextHandler.setRenderingContext(currentContext);
        }
        return url;
    }

}
