/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.documents.generators;

import java.io.IOException;
import java.util.Optional;
import java.util.Set;

import org.apache.avalon.framework.context.Context;
import org.apache.avalon.framework.context.ContextException;
import org.apache.avalon.framework.context.Contextualizable;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.search.SortOrder;
import org.ametys.cms.search.query.DocumentTypeQuery;
import org.ametys.cms.search.solr.SearcherFactory;
import org.ametys.cms.search.solr.SearcherFactory.SortDefinition;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.user.UserManager;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.LambdaUtils;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.indexing.solr.SolrWorkspacesConstants;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.ProjectsCatalogueManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.repository.page.Page;

/**
 * Document stream generator for the simple service.
 */
public class DocumentStreamGenerator extends ServiceableGenerator implements Contextualizable
{
    /** The project manager */
    protected ProjectManager _projectManager;
    
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    
    /** The project catalog helper */
    protected ProjectsCatalogueManager _projectCatalogManager;
    
    /** The document manager */
    protected DocumentWorkspaceModule _documentModule;
    
    /** Current user provider */
    protected CurrentUserProvider _currentUserProvider;
    
    /** The searcher factory */
    protected SearcherFactory _searcherFactory;
    
    /** The user manager */
    protected UserManager _userManager;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The right manager */
    protected RightManager _rightManager;
    
    /** Avalon context */
    protected Context _context;
    
    @Override
    public void contextualize(Context context) throws ContextException
    {
        _context = context;
    }
    
    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) serviceManager.lookup(ProjectMemberManager.ROLE);
        _projectCatalogManager = (ProjectsCatalogueManager)  serviceManager.lookup(ProjectsCatalogueManager.ROLE);
        WorkspaceModuleExtensionPoint moduleManagerEP = (WorkspaceModuleExtensionPoint) serviceManager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _documentModule = moduleManagerEP.getModule(DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
        _currentUserProvider = (CurrentUserProvider) serviceManager.lookup(CurrentUserProvider.ROLE);
        _searcherFactory = (SearcherFactory) serviceManager.lookup(SearcherFactory.ROLE);
        _userManager = (UserManager) serviceManager.lookup(UserManager.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        int max = parameters.getParameterAsInteger("max-results", 0);
        
        contentHandler.startDocument();
        XMLUtils.startElement(contentHandler, "documents");
        
        UserIdentity user = _currentUserProvider.getUser();
        
        if (user != null)
        {
            _getLastDocuments(user, max).stream().forEach(this::_saxDocument);
        }
        
        XMLUtils.endElement(contentHandler, "documents");
        contentHandler.endDocument();
    }

    private AmetysObjectIterable<Resource> _getLastDocuments(UserIdentity user, int max)
    {
        try
        {
            return _searcherFactory.create()
                .addFilterQuery(new DocumentTypeQuery(SolrWorkspacesConstants.TYPE_PROJECT_RESOURCE))
                .setCheckRights(true)
                .withLimits(0, max > 0 ? max : Integer.MAX_VALUE)
                .withSort(new SortDefinition("date-for-sorting", SortOrder.DESC))
                .search();
        }
        catch (Exception e)
        {
            throw new RuntimeException(String.format("An error occurred while requesting last documents for user '%s'.", user), e);
        }
    }
    
    /**
     * SAX necessary document properties for the simple document stream service.
     * @param resource The resource
     */
    protected void _saxDocument(Resource resource)
    {
        try
        {
            Project project = _projectManager.getParentProject(resource);
            ResourceCollection folder = resource.getParent();
            
            AttributesImpl attrs = new AttributesImpl();
            
            attrs.addCDATAAttribute("id", resource.getId());
            XMLUtils.startElement(contentHandler, "document", attrs);
            
            XMLUtils.createElement(contentHandler, "name", resource.getName());
           
            attrs.clear();
            attrs.addCDATAAttribute("id", folder.getId());
            String documentUrl = _getDocumentUrl(project, folder);
            attrs.addCDATAAttribute("url", documentUrl);
            XMLUtils.createElement(contentHandler, "folder", attrs, folder.getName());
            
            attrs.clear();
            attrs.addCDATAAttribute("id", project.getId());
            Optional<UserIdentity> creator = Optional.ofNullable(resource.getCreator());
            attrs.addCDATAAttribute("hasAccess", String.valueOf(creator.isPresent() && _projectMemberManager.isProjectMember(project, creator.get())));
            attrs.addCDATAAttribute("url", _projectManager.getProjectUrl(project, StringUtils.EMPTY));
            XMLUtils.createElement(contentHandler, "project", attrs, project.getTitle());
            
            _projectCatalogManager.saxCategory(contentHandler, project, "projectCategory");
            
            // Last modification: date and user
            XMLUtils.createElement(contentHandler, "lastModified", DateUtils.dateToString(resource.getLastModified()));
            
            creator.map(_userManager::getUser)
                .ifPresent(LambdaUtils.wrapConsumer(lastContributor -> _userHelper.saxUser(lastContributor, contentHandler)));
            
            XMLUtils.endElement(contentHandler, "document");
        }
        catch (SAXException e)
        {
            throw new RuntimeException("An error occurred while gathering the documents' information.");
        }
    }
    
    /**
     * Retrieves the URL of a folder in the document module of a project
     * @param project The project
     * @param folder The folder
     * @return The url or null if no module page is found for this project and the current language
     */
    protected String _getDocumentUrl(Project project, ResourceCollection folder)
    {
        Set<Page> documentModulePages = _projectManager.getModulePages(project, _documentModule);
        if (!documentModulePages.isEmpty())
        {
            Page documentModulePage = documentModulePages.iterator().next();
            StringBuilder sb = new StringBuilder();
            sb.append(ResolveURIComponent.resolve("page", documentModulePage.getId()));
            sb.append("#").append(folder.getId());
            return sb.toString();
        }
        
        return null;
    }
}
