/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.documents.onlyoffice;

import java.io.ByteArrayOutputStream;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.Path;
import java.nio.file.StandardCopyOption;
import java.security.GeneralSecurityException;
import java.util.Base64;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;

import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.impl.URLSource;
import org.apache.http.client.config.RequestConfig;
import org.apache.http.client.methods.CloseableHttpResponse;
import org.apache.http.client.methods.HttpPost;
import org.apache.http.entity.StringEntity;
import org.apache.http.impl.client.CloseableHttpClient;
import org.apache.http.impl.client.HttpClientBuilder;

import org.ametys.cms.content.indexing.solr.SolrResourceGroupedMimeTypes;
import org.ametys.core.authentication.token.AuthenticationTokenManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.ui.Callable;
import org.ametys.core.user.CurrentUserProvider;
import org.ametys.core.user.UserIdentity;
import org.ametys.core.util.JSONUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.explorer.resources.ResourceCollection;
import org.ametys.plugins.explorer.rights.ResourceRightAssignmentContext;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.WorkspacesHelper.FileType;
import org.ametys.plugins.workspaces.documents.WorkspaceExplorerResourceDAO;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.authentication.AccessDeniedException;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;
import org.ametys.runtime.util.AmetysHomeHelper;

/**
 * Main helper for OnlyOffice edition
 */
public class OnlyOfficeManager extends AbstractLogEnabled implements Component, Serviceable
{
    /** The Avalon role */
    public static final String ROLE = OnlyOfficeManager.class.getName();
    
    /** The path for workspace cache */
    public static final String WORKSPACE_PATH_CACHE = "cache/workspaces";
    
    /** The path for thumbnail file */
    public static final String THUMBNAIL_FILE_PATH = "file-manager/thumbnail";
    
    private static final byte[] __JWT_HEADER_BYTES = "{\"alg\":\"HS256\",\"typ\":\"JWT\"}".getBytes(StandardCharsets.UTF_8);
    private static final String __JWT_HEADER_BASE64 = Base64.getUrlEncoder().withoutPadding().encodeToString(__JWT_HEADER_BYTES);
    
    /** The token manager */
    protected AuthenticationTokenManager _tokenManager;
    /** The current user provider */
    protected CurrentUserProvider _currentUserProvider;
    /** The Ametys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The Only Office key manager */
    protected OnlyOfficeKeyManager _onlyOfficeKeyManager;
    /** The JSON utils */
    protected JSONUtils _jsonUtils;
    /** The source resolver */
    protected SourceResolver _sourceResolver;
    /** The documents module DAO */
    protected WorkspaceExplorerResourceDAO _workspaceExplorerResourceDAO;
    /** The rights manager */
    protected RightManager _rightManager;
    
    private Map<String, Lock> _locks = new ConcurrentHashMap<>();
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        _currentUserProvider = (CurrentUserProvider) manager.lookup(CurrentUserProvider.ROLE);
        _tokenManager = (AuthenticationTokenManager) manager.lookup(AuthenticationTokenManager.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _onlyOfficeKeyManager = (OnlyOfficeKeyManager) manager.lookup(OnlyOfficeKeyManager.ROLE);
        _jsonUtils = (JSONUtils) manager.lookup(JSONUtils.ROLE);
        _sourceResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _workspaceExplorerResourceDAO = (WorkspaceExplorerResourceDAO) manager.lookup(WorkspaceExplorerResourceDAO.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
    }
    
    /**
     * Determines if OnlyOffice edition is available
     * @return true if OnlyOffice edition is available
     */
    public boolean isOnlyOfficeAvailable()
    {
        return Config.getInstance().getValue("workspaces.onlyoffice.enabled", false, false);
    }
    
    /**
     * Get the needed information for Only Office edition
     * @param resourceId the id of resource to edit
     * @return the only office informations
     */
    @Callable (rights = Callable.READ_ACCESS, paramIndex = 0, rightContext = ResourceRightAssignmentContext.ID)
    public Map<String, Object> getOnlyOfficeInfo(String resourceId)
    {
        Map<String, Object> infos = new HashMap<>();
        
        OnlyOfficeResource resource = _getOnlyOfficeResource(resourceId, _currentUserProvider.getUser());
        
        Map<String, Object> fileInfo = new HashMap<>();
        fileInfo.put("title", resource.title());
        fileInfo.put("fileExtension", resource.fileExtension());
        fileInfo.put("key", resource.key());
        fileInfo.put("previewKey", resource.previewKey());
        fileInfo.put("urlDownload", resource.urlDownload());

        infos.put("file", fileInfo);
        infos.put("callbackUrl", resource.callbackUrl());
        
        return infos;
    }
    
    
    /**
     * Generate a token for OnlyOffice use
     * @param fileId id of the resource that will be used by OnlyOffice
     * @return the token
     */
    public String generateToken(String fileId)
    {
        return _generateToken(fileId, _currentUserProvider.getUser());
    }
    
    private String _generateToken(String fileId, UserIdentity user)
    {
        Set<String> contexts = Set.of(StringUtils.substringAfter(fileId, "://"));
        return _tokenManager.generateToken(user, 30000, true, null, contexts, "onlyOfficeResponse", null);
    }
    
    /**
     * Sign a json configuration for OnlyOffice using a secret parametrized key
     * @param toSign The json to sign
     * @return The signed json
     */
    @Callable (rights = Callable.CHECKED_BY_IMPLEMENTATION)
    public Map<String, Object> signConfiguration(String toSign)
    {
        Project project = _workspaceExplorerResourceDAO.getProjectFromRequest();

        ResourceCollection documentRoot = _workspaceExplorerResourceDAO.getRootFromProject(project);

        if (!_rightManager.currentUserHasReadAccess(documentRoot))
        {
            throw new AccessDeniedException("User '" + _currentUserProvider.getUser() + "' tried to do read operation without convenient right");
        }
        
        Map<String, Object> result = new HashMap<>();
        
        String token;
        try
        {
            token = _signConfiguration(toSign);
            
            if (StringUtils.isNotBlank(token))
            {
                result.put("signature", token);
            }
            
            result.put("success", "true");
            return result;
        }
        catch (GeneralSecurityException e)
        {
            result.put("success", "false");
            return result;
        }
    }
    
    private String _signConfiguration(String toSign) throws GeneralSecurityException
    {
        String secret = Config.getInstance().getValue("workspaces.onlyoffice.secret");
        
        if (StringUtils.isNotBlank(secret))
        {
            byte[] payloadBytes = toSign.getBytes(StandardCharsets.UTF_8);
            byte[] secretBytes = secret.getBytes(StandardCharsets.UTF_8);
            
            String payload = Base64.getUrlEncoder().withoutPadding().encodeToString(payloadBytes);
            
            String signingInput = __JWT_HEADER_BASE64 + "." + payload;
            byte[] signingInputBytes = signingInput.getBytes(StandardCharsets.UTF_8);

            String algorithm = "HmacSHA256";
            Mac hmac = Mac.getInstance(algorithm);
            hmac.init(new SecretKeySpec(secretBytes, algorithm));
            byte[] signatureBytes = hmac.doFinal(signingInputBytes);

            String signature = Base64.getUrlEncoder().withoutPadding().encodeToString(signatureBytes);

            String token = String.format("%s.%s.%s", __JWT_HEADER_BASE64, payload, signature);
            
            return token;
        }
        
        return null;
    }
    
    /**
     * Determines if the resource file can have a preview of thumbnail from only office
     * @param resourceId the resource id
     * @return <code>true</code> if resource file can have a preview of thumbnail from only office
     */
    public boolean canBePreviewed(String resourceId)
    {
        if (!isOnlyOfficeAvailable())
        {
            return false;
        }
        
        Resource resource = _resolver.resolveById(resourceId);
        
        List<FileType> allowedFileTypes = List.of(
                FileType.PDF,
                FileType.PRES,
                FileType.SPREADSHEET,
                FileType.TEXT
        );
        
        return SolrResourceGroupedMimeTypes.getGroup(resource.getMimeType())
            .map(groupMimeType -> allowedFileTypes.contains(FileType.valueOf(groupMimeType.toUpperCase())))
            .orElse(false);
    }
    
    /**
     * Generate thumbnail of the resource as png
     * @param projectName the project name
     * @param resourceId the resource id
     * @param user the user generating the thumbnail
     * @return <code>true</code> is the thumbnail is generated
     */
    public boolean generateThumbnailInCache(String projectName, String resourceId, UserIdentity user)
    {
        Lock lock = _locks.computeIfAbsent(resourceId, __ -> new ReentrantLock());
        lock.lock();
        
        try
        {
            File thumbnailFile = getThumbnailFile(projectName, resourceId);
            if (thumbnailFile != null && thumbnailFile.exists())
            {
                return true;
            }
        
            if (canBePreviewed(resourceId))
            {
                String urlPrefix = Config.getInstance().getValue("workspaces.onlyoffice.server.url");
                String url = StringUtils.stripEnd(urlPrefix, "/") + "/ConvertService.ashx";
                
                RequestConfig requestConfig = RequestConfig.custom()
                        .setConnectTimeout(30000)
                        .setSocketTimeout(30000)
                        .build();
                try (CloseableHttpClient httpclient = HttpClientBuilder.create()
                                                                       .setDefaultRequestConfig(requestConfig)
                                                                       .useSystemProperties()
                                                                       .build())
                {
                    // Prepare a request object
                    HttpPost post = new HttpPost(url);
                    OnlyOfficeResource resource = _getOnlyOfficeResource(resourceId, user);
                    
                    Map<String, Object> thumbnailParameters = new HashMap<>();
                    thumbnailParameters.put("outputtype", "png");
                    thumbnailParameters.put("filetype", resource.fileExtension());
                    thumbnailParameters.put("key", resource.key());
                    thumbnailParameters.put("previewkey", resource.previewKey()); // TODO
                    thumbnailParameters.put("url", resource.urlDownload());
                    
                    Map<String, Object> sizeInputs = new HashMap<>();
                    sizeInputs.put("aspect", 1);
                    sizeInputs.put("height", 1000);
                    sizeInputs.put("width", 300);
                    
                    thumbnailParameters.put("thumbnail", sizeInputs);
                    
                    String jsonBody = _jsonUtils.convertObjectToJson(thumbnailParameters);
                    StringEntity params = new StringEntity(jsonBody);
                    post.addHeader("content-type", "application/json");
                    
                    String jwtToken = _signConfiguration(jsonBody);
                    if (jwtToken != null)
                    {
                        post.addHeader("Authorization", "Bearer " + jwtToken);
                    }
                    
                    post.setEntity(params);
                    
                    try (CloseableHttpResponse httpResponse = httpclient.execute(post))
                    {
                        int statusCode = httpResponse.getStatusLine().getStatusCode();
                        if (statusCode != 200)
                        {
                            getLogger().error("An error occurred getting thumbnail for resource id '{}'. HTTP status code response is '{}'", resourceId, statusCode);
                            return false;
                        }
                        
                        ByteArrayOutputStream bos = new ByteArrayOutputStream();
                        try (InputStream is = httpResponse.getEntity().getContent())
                        {
                            IOUtils.copy(is, bos);
                        }
                        
                        String responseAsStringXML = bos.toString();
                        if (responseAsStringXML.contains("<Error>"))
                        {
                            String errorMsg = StringUtils.substringBefore(StringUtils.substringAfter(responseAsStringXML, "<Error>"), "</Error>");
                            getLogger().error("An error occurred getting thumbnail for resource id '{}'. Error message is '{}'", resourceId, errorMsg);
                            return false;
                        }
                        else
                        {
                            String previewURL = StringUtils.substringBefore(StringUtils.substringAfter(responseAsStringXML, "<FileUrl>"), "</FileUrl>");
                            String decodeURL = StringUtils.replace(previewURL, "&amp;", "&");
                            
                            _generatePNGFileInCache(projectName, decodeURL, resourceId);
                            
                            return true;
                        }
                    }
                    catch (Exception e)
                    {
                        getLogger().error("Error getting thumbnail for file {}", resource.title(), e);
                    }
                }
                catch (Exception e)
                {
                    getLogger().error("Unable to contact Only Office conversion API to get thumbnail.", e);
                }
            }
            
            return false;
        }
        finally
        {
            lock.unlock();
            _locks.remove(resourceId, lock); // possible minor race condition here, with no effect if the thumbnail has been correctly generated
        }
    }
    
    /**
     * Delete thumbnail in cache
     * @param projectName the project name
     * @param resourceId the resourceId id
     */
    public void deleteThumbnailInCache(String projectName, String resourceId)
    {
        try
        {
            File file = getThumbnailFile(projectName, resourceId);
            if (file != null && file.exists())
            {
                FileUtils.forceDelete(file);
            }
        }
        catch (Exception e)
        {
            getLogger().error("Can delete thumbnail in cache for project name '{}' and resource id '{}'", projectName, resourceId, e);
        }
    }
    
    /**
     * Delete project thumbnails in cache
     * @param projectName the project name
     */
    public void deleteProjectThumbnailsInCache(String projectName)
    {
        try
        {
            File thumbnailDir = new File(AmetysHomeHelper.getAmetysHomeData(), WORKSPACE_PATH_CACHE + "/" + projectName);
            if (thumbnailDir.exists())
            {
                FileUtils.forceDelete(thumbnailDir);
            }
        }
        catch (Exception e)
        {
            getLogger().error("Can delete thumbnails in cache for project name '{}'", projectName, e);
        }
    }
    
    /**
     * Generate a png file from the uri
     * @param projectName the project name
     * @param uri the uri
     * @param fileId the id of the file
     * @throws IOException if an error occurred
     */
    protected void _generatePNGFileInCache(String projectName, String uri, String fileId) throws IOException
    {
        Path thumbnailDir = AmetysHomeHelper.getAmetysHomeData().toPath().resolve(Path.of(WORKSPACE_PATH_CACHE, projectName, THUMBNAIL_FILE_PATH));
        Files.createDirectories(thumbnailDir);
        
        String name = _encodeFileId(fileId);
       
        URLSource source = null;
        Path tmpFile = thumbnailDir.resolve(name + ".tmp.png");
        try
        {
            // Resolve the export to the appropriate png url.
            source = (URLSource) _sourceResolver.resolveURI(uri, null, new HashMap<>());
           
            // Save the preview image into a temporary file.
            try (InputStream is = source.getInputStream(); OutputStream os = Files.newOutputStream(tmpFile))
            {
                IOUtils.copy(is, os);
            }
           
            // If all went well until now, rename the temporary file
            Files.move(tmpFile, tmpFile.resolveSibling(name + ".png"), StandardCopyOption.REPLACE_EXISTING);
        }
        catch (Exception e)
        {
            getLogger().error("An error occurred generating png file with uri '{}'", uri, e);
        }
        finally
        {
            if (source != null)
            {
                _sourceResolver.release(source);
            }
        }
    }
    
    private OnlyOfficeResource _getOnlyOfficeResource(String resourceId, UserIdentity user)
    {
        Resource resource = _resolver.resolveById(resourceId);
        
        String token = _generateToken(resourceId, user);
        String tokenCtx = StringUtils.substringAfter(resourceId, "://");
        
        String title = resource.getName();
        String fileExtension = StringUtils.substringAfterLast(resource.getName(), ".").toLowerCase();
        String key = _onlyOfficeKeyManager.getKey(resourceId);
        String previewKey = tokenCtx + "." + Optional.ofNullable(resource.getLastModified()).map(Date::getTime).orElse(0L);
        
        String ooCMSUrl = Config.getInstance().getValue("workspaces.onlyoffice.bo.url");
        if (StringUtils.isEmpty(ooCMSUrl))
        {
            ooCMSUrl = Config.getInstance().getValue("cms.url");
        }
        
        String downloadUrl = ooCMSUrl
                + "/_workspaces/only-office/download-resource?"
                + "id=" + resourceId
                + "&token=" + token
                + "&tokenContext=" + tokenCtx;
        
        String callbackUrl = ooCMSUrl
                + "/_workspaces/only-office/response.json?"
                + "id=" + resourceId
                + "&token=" + token
                + "&tokenContext=" + tokenCtx;
        
        return new OnlyOfficeResource(title, fileExtension, key, previewKey, downloadUrl, callbackUrl);
    }
    
    /**
     * Get thumbnail file
     * @param projectName the project name
     * @param resourceId the resource id
     * @return the thumbnail file. Can be <code>null</code> if doesn't exist.
     */
    public File getThumbnailFile(String projectName, String resourceId)
    {
        File thumbnailDir = new File(AmetysHomeHelper.getAmetysHomeData(), WORKSPACE_PATH_CACHE + "/" + projectName + "/" + THUMBNAIL_FILE_PATH);
        if (thumbnailDir.exists())
        {
            String name = _encodeFileId(resourceId);
            return new File(thumbnailDir, name + ".png");
        }
        
        return null;
    }
    
    private String _encodeFileId(String fileId)
    {
        return Base64.getEncoder().withoutPadding().encodeToString(fileId.getBytes(StandardCharsets.UTF_8));
    }
    
    private record OnlyOfficeResource(String title, String fileExtension, String key, String previewKey, String urlDownload, String callbackUrl) { }
}
