/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.files;

import java.time.ZonedDateTime;
import java.util.HashMap;
import java.util.Map;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.query.Query;

import org.apache.avalon.framework.parameters.Parameters;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.acting.ServiceableAction;
import org.apache.cocoon.environment.Redirector;
import org.apache.cocoon.environment.SourceResolver;

import org.ametys.core.util.DateUtils;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.QueryHelper;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.documents.jcr.File;
import org.ametys.plugins.workspaces.documents.jcr.File.PublicLinkToken;
import org.ametys.runtime.authentication.AccessDeniedException;

/**
 * Check if a public token allows to download a File.
 */
public class CheckWorkspacesFileDownloadPublicTokenReadAccessAction extends ServiceableAction
{
    /** The Ametys object resolver. */
    protected AmetysObjectResolver _resolver;
    
    private Repository _repository;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _repository = (Repository) smanager.lookup(AbstractRepository.ROLE);
    }
    
    @Override
    public Map act(Redirector redirector, SourceResolver resolver, Map objectModel, String source, Parameters parameters) throws Exception
    {
        String tokenId = parameters.getParameter("tokenId", "");
        PublicLinkToken token = _getTokenByID(tokenId);
        
        if (token == null)
        {
            throw new AccessDeniedException("CheckPublicTokenReadAccessAction: a valid token must be provided.");
        }
        // Get the Ametys object from the token
        File file = _resolver.resolveById(token.fileId());
        Map<String, String> result = new HashMap<>();
        result.put("filePath", file.getPath());
        
        return result;
    }

    private PublicLinkToken _getTokenByID(String tokenId)
    {
        Session session = null;
        try
        {
            session = _repository.login();

            StringExpression tokenExpression = new StringExpression(File.TOKEN_ID, Operator.EQ, tokenId);
            String jcrQuery = QueryHelper.getXPathQuery(null, "ametys:compositeMetadata", tokenExpression);
            
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(jcrQuery, Query.XPATH);
            NodeIterator nodes = query.execute().getNodes();
            
            while (nodes.hasNext())
            {
                Node node = (Node) nodes.next();
                String fileId = node.getProperty(File.TOKEN_FILE_ID_PROPERTY).getString();
                ZonedDateTime date = DateUtils.asZonedDateTime(node.getProperty(File.TOKEN_CREATION_DATE_PROPERTY).getDate());

                // check if node parent parent id is file with correct file id
                Node tokensNode = node.getParent();
                Node fileNode = tokensNode.getParent();
                String fileNodeId = "file://" + fileNode.getIdentifier();
                if (fileNodeId.equals(fileId))
                {
                    return new File.PublicLinkToken(tokenId, date, fileId);
                }
            }
            
            return null;
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occurred executing the JCR query to get token " + tokenId, ex);
        }
    }
}
