/*
 *  Copyright 2023 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.forum;

import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.cms.data.Binary;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

import com.google.common.collect.ImmutableSet;

/**
 * Workspaces module for forums
 */
public class ForumWorkspaceModule extends AbstractWorkspaceModule
{

    /** The id of forum module */
    public static final String FORUM_MODULE_ID = ForumWorkspaceModule.class.getName();
    
    /** The service id of forum module */
    public static final String FORUM_SERVICE_ID = "org.ametys.plugins.workspaces.module.Forums";
    
    /** Workspaces forums node name */
    private static final String __WORKSPACES_FORUMS_NODE_NAME = "forums";

    private static final String __FORUM_NUMBER_HEADER_ID = __WORKSPACES_FORUMS_NODE_NAME + "$forum_number";

    /** Workspace threads DAO */
    private WorkspaceThreadDAO _workspaceThreadDAO;
    
    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _workspaceThreadDAO = (WorkspaceThreadDAO) manager.lookup(WorkspaceThreadDAO.ROLE);
    }
    
    @Override
    public String getId()
    {
        return FORUM_MODULE_ID;
    }

    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_FORUM_LABEL");
    }

    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_FORUM_DESCRIPTION");
    }

    public String getModuleName()
    {
        return __WORKSPACES_FORUMS_NODE_NAME;
    }

    public int getOrder()
    {
        return ORDER_FORUMS;
    }

    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_THREAD_CREATED, 
                ObservationConstants.EVENT_THREAD_MODIFIED,
                ObservationConstants.EVENT_THREAD_CLOSED,
                ObservationConstants.EVENT_THREAD_DELETED,
                ObservationConstants.EVENT_THREAD_COMMENTED, 
                ObservationConstants.EVENT_THREAD_ACCEPTED);
    }


    @Override
    protected String getModulePageName()
    {
        return "forum";
    }

    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_NEW_FORUM_TITLE");
    }

    @Override
    protected void initializeModulePage(ModifiablePage forumPage)
    {
        ModifiableZone defaultZone = forumPage.createZone("default");
        
        String serviceId = FORUM_SERVICE_ID;
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.SERVICE);
        defaultZoneItem.setServiceId(serviceId);
        
        ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
        serviceDataHolder.setValue("xslt", _getDefaultXslt(serviceId));
    }

    /**
     * Get the URI of a thread in project'site
     * @param project The project
     * @param threadId The id of thread
     * @return The thread uri
     */
    public String getThreadUri(Project project, String threadId)
    {
        String moduleUrl = getModuleUrl(project);
        if (moduleUrl != null)
        {
            StringBuilder sb = new StringBuilder();
            sb.append(moduleUrl);
            sb.append("#thread-").append(threadId);
            
            return sb.toString();
        }
        
        return null;
    }
    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            return Map.of(__FORUM_NUMBER_HEADER_ID, _workspaceThreadDAO.getThreadsCount(project));
        }
        else
        {
            return Map.of(__FORUM_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }

    @Override
    public long getModuleSize(Project project)
    {
        return _workspaceThreadDAO.getProjectThreads(project, false)
                         .stream()
                         .map(Thread::getAttachments)
                         .flatMap(Collection::stream)
                         .mapToLong(Binary::getLength)
                         .sum();
    }
    
    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__FORUM_NUMBER_HEADER_ID, new I18nizableText("plugin.workspaces", "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_FORUM_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }
}
