/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.util.Arrays;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.stream.Collectors;

import org.apache.avalon.framework.component.Component;
import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.cms.content.ContentHelper;
import org.ametys.cms.data.ContentValue;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.cms.search.content.ContentSearcherFactory.SimpleContentSearcher;
import org.ametys.cms.search.query.ContentQuery;
import org.ametys.cms.search.query.NotQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.runtime.plugin.component.AbstractLogEnabled;

/**
 * Helper for members
 *
 */
public class MemberHelper extends AbstractLogEnabled implements Serviceable, Component
{
    /** The Avalon role*/
    public static final String ROLE = MemberHelper.class.getName();
    
    /** The content searcher */
    protected ContentSearcherFactory _contentSearcherFactory;
    /** The AMetys object resolver */
    protected AmetysObjectResolver _resolver;
    /** The content helper */
    protected ContentHelper _contentHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        _resolver = (AmetysObjectResolver) smanager.lookup(AmetysObjectResolver.ROLE);
        _contentHelper = (ContentHelper) smanager.lookup(ContentHelper.ROLE);
        _contentSearcherFactory = (ContentSearcherFactory) smanager.lookup(ContentSearcherFactory.ROLE);
    }
    
    /**
     * Get the skills' id of a member
     * @param content the content
     * @return the skills' id.
     */
    public List<String> getSkills(Content content)
    {
        if (content.hasValue("skills"))
        {
            ContentValue[] skillValues = content.getValue("skills", false, new ContentValue[0]);
            
            if (skillValues.length != 0)
            {
                return Arrays.stream(skillValues)
                    .map(v -> v.getContentId())
                    .collect(Collectors.toList());
            }
        }
        
        return List.of();
    }
    
    /**
     * Get the keywords' id of a member
     * @param content the content
     * @return the keywords' id.
     */
    public List<String> getKeywords(Content content)
    {
        if (content.hasValue("keywords"))
        {
            ContentValue[] keywordValues = content.getValue("keywords", false, new ContentValue[0]);
            
            if (keywordValues.length != 0)
            {
                return Arrays.stream(keywordValues)
                    .map(v -> v.getContentId())
                    .collect(Collectors.toList());
            }
        }
        
        return List.of();
    }
    
    /**
     * Get the related members based on common skills and keywords 
     * @param content the current member content
     * @param max the max number of related members
     * @return the related members
     * @throws Exception if search failed
     */
    public List<Content> getRelatedMembers(Content content, int max) throws Exception
    {
        List<String> skills = getSkills(content);
        List<String> keywords = getKeywords(content);
        
        if (!skills.isEmpty() || !keywords.isEmpty())
        {
            SimpleContentSearcher searcher = _contentSearcherFactory.create(WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
            
            // search user content with at least one common skill or one common keyword
            // results will be sorted by score, so by the number of common skills and keywords
            
            Collection<Query> queries = new HashSet<>();
            for (String skillId : skills)
            {
                Query skillQuery = new ContentQuery("skills", Operator.EQ, skillId, _resolver, _contentHelper);
                queries.add(skillQuery);
            }
            
            for (String keywordId : keywords)
            {
                Query keywordQuery = new ContentQuery("keywords", Operator.EQ, keywordId, _resolver, _contentHelper);
                queries.add(keywordQuery);
            }
            
            
            AmetysObjectIterable<Content> relatedMembers = searcher
                    .withFilterQueries(List.of(new NotQuery(() -> "id:\"" + content.getId() + "\""))) // exclude current content
                    .withLimits(0, max)
                    .search(new OrQuery(queries));
            
            return relatedMembers.stream()
                    .collect(Collectors.toList());
        }
        
        return List.of();
    }

}
