/*
 *  Copyright 2016 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.util.ArrayList;
import java.util.Date;
import java.util.GregorianCalendar;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;

import javax.jcr.Node;
import javax.jcr.NodeIterator;
import javax.jcr.PathNotFoundException;
import javax.jcr.Repository;
import javax.jcr.RepositoryException;
import javax.jcr.Session;
import javax.jcr.Value;
import javax.jcr.ValueFormatException;
import javax.jcr.query.Query;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.explorer.resources.jcr.JCRResourcesCollection;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.RepositoryConstants;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.repository.jcr.NodeTypeHelper;
import org.ametys.plugins.repository.provider.AbstractRepository;
import org.ametys.plugins.repository.query.SortCriteria;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModuleExtensionPoint;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.ZoneItem.ZoneType;

import com.google.common.collect.ImmutableSet;

/**
 * Helper component for managing members
 */
public class MembersWorkspaceModule extends AbstractWorkspaceModule
{
    /** The id of members module */
    public static final String MEMBERS_MODULE_ID = MembersWorkspaceModule.class.getName();
    
    /** Id of service of members */
    public static final String MEMBERS_SERVICE_ID = "org.ametys.plugins.workspaces.module.Members";
    
    /** Workspaces members node name */
    private static final String __WORKSPACES_MEMBERS_NODE_NAME = "members";
    
    private static final String __INVITATIONS_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":invitations";
    
    private static final String __INVITATION_NODE_NAME = RepositoryConstants.NAMESPACE_PREFIX + ":invitation";
    
    private static final String __NODETYPE_INVITATIONS = RepositoryConstants.NAMESPACE_PREFIX + ":invitations";
    
    private static final String __NODETYPE_INVITATION = RepositoryConstants.NAMESPACE_PREFIX + ":invitation";

    private static final String __MEMBER_NUMBER_HEADER_ID = __WORKSPACES_MEMBERS_NODE_NAME + "$member_number";
    
    /** Constants for invitation's date property */
    private static final String __INVITATION_DATE = RepositoryConstants.NAMESPACE_PREFIX + ":date";
    /** Constants for invitation's mail property */
    private static final String __INVITATION_MAIL = RepositoryConstants.NAMESPACE_PREFIX + ":mail";
    /** Constants for invitation's author property */
    private static final String __INVITATION_AUTHOR = RepositoryConstants.NAMESPACE_PREFIX + ":author";
    /** Constants for invitation's acl node */
    private static final String __INVITATION_ACL = RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":temp-acl";

    private WorkspaceModuleExtensionPoint _moduleEP;
    
    private Repository _repository;
    
    private ProjectMemberManager _projectMemberManager;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _moduleEP = (WorkspaceModuleExtensionPoint) smanager.lookup(WorkspaceModuleExtensionPoint.ROLE);
        _repository = (Repository) smanager.lookup(AbstractRepository.ROLE);
        _projectMemberManager = (ProjectMemberManager) smanager.lookup(ProjectMemberManager.ROLE);
    }
    
    @Override
    public String getId()
    {
        return MEMBERS_MODULE_ID;
    }
    
    public int getOrder()
    {
        return ORDER_MEMBERS;
    }
    
    public String getModuleName()
    {
        return __WORKSPACES_MEMBERS_NODE_NAME;
    }
    
    @Override
    protected String getModulePageName()
    {
        return "members";
    }
    
    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_MEMBERS_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_SERVICE_MODULE_MEMBERS_DESCRIPTION");
    }
    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_MEMBERS_TITLE");
    }
    
    @Override
    protected void initializeModulePage(ModifiablePage memberPage)
    {
        ModifiableZone defaultZone = memberPage.createZone("default");
        
        ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
        defaultZoneItem.setType(ZoneType.SERVICE);
        defaultZoneItem.setServiceId(MEMBERS_SERVICE_ID);
        
        ModifiableModelAwareDataHolder params = defaultZoneItem.getServiceParameters();
        
        params.setValue("header", _i18nUtils.translate(getModulePageTitle(), memberPage.getSitemapName()));
        // params.setValue("expandGroup", false); FIXME new service
        // params.setValue("nbMembers", -1); FIXME new service
        params.setValue("xslt", _getDefaultXslt(MEMBERS_SERVICE_ID));
    }

    @Override
    public Set<String> getAllowedEventTypes()
    {
        return ImmutableSet.of(ObservationConstants.EVENT_MEMBER_ADDED);
    }
    
    // -------------------------------------------------
    //                  INVITATIONS
    // -------------------------------------------------
    
    /**
     * Get the node holding the invitations
     * @param project The project
     * @param create true to create the node if it does not exist
     * @return the invitations' root node
     * @throws RepositoryException if an error occurred
     */
    public Node getInvitationsRootNode(Project project, boolean create) throws RepositoryException
    {
        ModifiableResourceCollection moduleRoot = getModuleRoot(project, create);
        
        Node moduleNode = ((JCRResourcesCollection) moduleRoot).getNode();
        Node node = null;
        
        if (moduleNode.hasNode(__INVITATIONS_NODE_NAME))
        {
            node = moduleNode.getNode(__INVITATIONS_NODE_NAME);
        }
        else if (create)
        {
            node = moduleNode.addNode(__INVITATIONS_NODE_NAME, __NODETYPE_INVITATIONS);
            moduleNode.getSession().save();
        }
        
        return node;
    }
    
    /**
     * Get the invitations for a given mail
     * @param email the email
     * @return the invitations
     */
    public List<Invitation> getInvitations(String email)
    {
        Session session = null;
        try
        {
            session = _repository.login();
            
            Expression expr = new StringExpression("mail", Operator.EQ, email);
            String xPathQuery = getInvitationXPathQuery(null, expr, null);
            
            @SuppressWarnings("deprecation")
            Query query = session.getWorkspace().getQueryManager().createQuery(xPathQuery, Query.XPATH);
            NodeIterator nodes = query.execute().getNodes();
     
            List<Invitation> invitations = new ArrayList<>();
            
            while (nodes.hasNext())
            {
                Node node = (Node) nodes.next();
                invitations.add(_getInvitation(node));
            }
            
            return invitations;
        }
        catch (RepositoryException ex)
        {
            if (session != null)
            {
                session.logout();
            }

            throw new AmetysRepositoryException("An error occurred executing the JCR query to get invitations for email " + email, ex);
        }
    }
    
    /**
     * Returns the invitation sorted by ascending date
     * @param project The project
     * @return The invitations node
     * @throws RepositoryException if an error occurred
     */
    public List<Invitation> getInvitations(Project project) throws RepositoryException
    {
        List<Invitation> invitations = new ArrayList<>();
        
        SortCriteria sortCriteria = new SortCriteria();
        sortCriteria.addJCRPropertyCriterion(__INVITATION_DATE, false, false);
        
        String xPathQuery = getInvitationXPathQuery(project, null, sortCriteria);
        
        @SuppressWarnings("deprecation")
        Query query = project.getNode().getSession().getWorkspace().getQueryManager().createQuery(xPathQuery, Query.XPATH);
        NodeIterator nodes = query.execute().getNodes();
        
        while (nodes.hasNext())
        {
            Node node = (Node) nodes.next();
            invitations.add(_getInvitation(node));
        }
        
        return invitations;
    }
    
    private Invitation _getInvitation(Node node) throws ValueFormatException, PathNotFoundException, RepositoryException
    {
        String email = node.getProperty(__INVITATION_MAIL).getString();
        Date date = node.getProperty(__INVITATION_DATE).getDate().getTime();
        Node authorNode = node.getNode(__INVITATION_AUTHOR);
        UserIdentity author = new UserIdentity(authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login").getString(), authorNode.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population").getString());
        
        Map<String, String> allowedProfileByModules = new HashMap<>();
        
        Node aclNode = node.getNode(RepositoryConstants.NAMESPACE_PREFIX_INTERNAL + ":temp-acl");
        NodeIterator children = aclNode.getNodes();
        while (children.hasNext())
        {
            Node child = (Node) children.next();
            if (child.hasProperty(RepositoryConstants.NAMESPACE_PREFIX + ":allowed-profiles"))
            {
                Value[] values = child.getProperty(RepositoryConstants.NAMESPACE_PREFIX + ":allowed-profiles").getValues();
                if (values.length > 0)
                {
                    String moduleName = child.getName();
                    WorkspaceModule module = _moduleEP.getModuleByName(moduleName);
                    if (module != null)
                    {
                        allowedProfileByModules.put(module.getId(), values[0].getString());
                    }
                }
            }
        }
        
        return new Invitation(email, date, author, allowedProfileByModules, _getProjectName(node));
    }
    
    private String _getProjectName(Node node)
    {
        try
        {
            Node parentNode = node.getParent();
            
            while (parentNode != null && !NodeTypeHelper.isNodeType(parentNode, "ametys:project"))
            {
                parentNode = parentNode.getParent();
            }
            
            return parentNode != null ? parentNode.getName() : null;
        }
        catch (RepositoryException e)
        {
            getLogger().error("Unable to get parent project", e);
            return null;
        }
    }
    
    /**
     * Creates the XPath query corresponding to specified {@link Expression}.
     * @param project The project. Can be null to browser all projects
     * @param invitExpression the query predicates.
     * @param sortCriteria the sort criteria.
     * @return the created XPath query.
     * @throws RepositoryException if an error occurred
     */
    public String getInvitationXPathQuery(Project project, Expression invitExpression, SortCriteria sortCriteria) throws RepositoryException
    {
        String predicats = null;
        
        if (invitExpression != null)
        {
            predicats = StringUtils.trimToNull(invitExpression.build());
        }
        
        StringBuilder xpathQuery = new StringBuilder();
        
        if (project != null)
        {
            xpathQuery.append("/jcr:root")
                .append(getInvitationsRootNode(project, true).getPath());
        }
        
        xpathQuery.append("//element(*, " + __NODETYPE_INVITATION + ")");
        
        if (predicats != null)
        {
            xpathQuery.append("[" + predicats + "]");
        }
        
        if (sortCriteria != null)
        {
            xpathQuery.append(" " + sortCriteria.build());
        }
        
        return xpathQuery.toString();
    }
    
    /**
     * Add an invitation
     * @param project The project
     * @param mail The mail
     * @param invitDate The invitation's date
     * @param author The invitation's author
     * @param allowedProfileByModules The allowed profiles by modules
     * @return The created invitation node
     * @throws RepositoryException if an error occurred
     */
    public Invitation addInvitation (Project project, Date invitDate, String mail, UserIdentity author, Map<String, String> allowedProfileByModules) throws RepositoryException
    {
        Node contextNode = getInvitationsRootNode(project, true);
        
        Node invitNode = contextNode.addNode(__INVITATION_NODE_NAME, __NODETYPE_INVITATION);
        
        // Date
        GregorianCalendar gc = new GregorianCalendar();
        gc.setTime(invitDate);
        
        invitNode.setProperty(__INVITATION_DATE, gc);
        
        // Mail
        invitNode.setProperty(__INVITATION_MAIL, mail);
        
        // Author
        Node authorNode = invitNode.addNode(__INVITATION_AUTHOR);
        authorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":login", author.getLogin());
        authorNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":population", author.getPopulationId());
        
        Node aclNode = invitNode.addNode(__INVITATION_ACL);
        
        for (Entry<String, String> allowedProfile : allowedProfileByModules.entrySet())
        {
            String moduleId = allowedProfile.getKey();
            String profileId = allowedProfile.getValue();
            
            WorkspaceModule module = _moduleEP.getExtension(moduleId);
            if (module != null)
            {
                Node moduleNode = aclNode.addNode(module.getModuleName());
                moduleNode.setProperty(RepositoryConstants.NAMESPACE_PREFIX + ":allowed-profiles", new String[] {profileId});
            }
        }
        
        return new Invitation(mail, invitDate, author, allowedProfileByModules, project.getName());
    }
    
    /**
     * Determines if a invitation already exists for this email
     * @param project the project
     * @param mail the mail to test
     * @return true if a invitation exists
     * @throws RepositoryException if an error occured
     */
    public boolean invitationExists(Project project, String mail) throws RepositoryException
    {
        Node contextNode = getInvitationsRootNode(project, true);
        
        NodeIterator nodes = contextNode.getNodes(__INVITATION_NODE_NAME);
        while (nodes.hasNext())
        {
            Node node = (Node) nodes.next();
            if (node.hasProperty(__INVITATION_MAIL) && node.getProperty(__INVITATION_MAIL).getString().equals(mail))
            {
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Remove a invitation 
     * @param project the project
     * @param mail the mail to remove
     * @return true if a invitation has been removed
     * @throws RepositoryException if an error occured
     */
    public boolean removeInvitation(Project project, String mail) throws RepositoryException
    {
        Node contextNode = getInvitationsRootNode(project, true);
        
        NodeIterator nodes = contextNode.getNodes(__INVITATION_NODE_NAME);
        while (nodes.hasNext())
        {
            Node node = (Node) nodes.next();
            if (node.hasProperty(__INVITATION_MAIL) && node.getProperty(__INVITATION_MAIL).getString().equals(mail))
            {
                node.remove();
                return true;
            }
        }
        
        return false;
    }
    
    /**
     * Bean representing a invitation by email
     *
     */
    public class Invitation
    {
        private Date _date;
        private UserIdentity _author;
        private String _email;
        private Map<String, String> _allowedProfileByModules;
        private String _projectName;
        
        /**
         * Constructor
         * @param email The email
         * @param date The date of invitation
         * @param author The author of invitation
         * @param allowedProfileByModules The rights
         * @param projectName the name of parent project
         */
        public Invitation(String email, Date date, UserIdentity author, Map<String, String> allowedProfileByModules, String projectName)
        {
            _email = email;
            _date = date;
            _author = author;
            _allowedProfileByModules = allowedProfileByModules;
            _projectName = projectName;
        }
        
        /**
         * Get the email
         * @return the email
         */
        public String getEmail()
        {
            return _email;
        }
        
        /**
         * Get the date of invitation
         * @return the date of invitation
         */
        public Date getDate()
        {
            return _date;
        }
        
        /**
         * Get the author of invitation
         * @return the author of invitation
         */
        public UserIdentity getAuthor()
        {
            return _author;
        }
        
        /**
         * Get the allowed profile for each module
         * @return the allowed profile for each module
         */
        public Map<String, String> getAllowedProfileByModules()
        {
            return _allowedProfileByModules;
        }
        
        /**
         * Get the parent project name
         * @return the parent project name
         */
        public String getProjectName()
        {
            return _projectName;
        }
        
        @Override
        public String toString()
        {
            return "Invitation [mail=" + _email + ", project=" + _projectName + "]";
        }
    }

    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, true, null);

            if (projectMembers != null)
            {
                return Map.of(__MEMBER_NUMBER_HEADER_ID, projectMembers.size());
            }
            else
            {
                return Map.of(__MEMBER_NUMBER_HEADER_ID, __SIZE_ERROR);
            }
        }
        else
        {
            return Map.of(__MEMBER_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }
    
    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__MEMBER_NUMBER_HEADER_ID, new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_MEMBERS"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    protected boolean _showActivatedStatus()
    {
        return false;
    }

    @Override
    public Set<String> getAllEventTypes()
    {
        return Set.of(ObservationConstants.EVENT_MEMBER_ADDED,
                      ObservationConstants.EVENT_MEMBER_DELETED);
    }
}

