/*
 *  Copyright 2018 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.ProcessingException;
import org.apache.cocoon.environment.ObjectModelHelper;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.generation.ServiceableGenerator;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.commons.lang3.ArrayUtils;
import org.apache.commons.lang3.LocaleUtils;
import org.xml.sax.SAXException;

import org.ametys.cms.contenttype.ContentType;
import org.ametys.cms.contenttype.ContentTypeExtensionPoint;
import org.ametys.cms.repository.Content;
import org.ametys.core.group.GroupManager;
import org.ametys.core.right.RightManager;
import org.ametys.core.user.User;
import org.ametys.core.user.UserIdentity;
import org.ametys.plugins.core.user.UserHelper;
import org.ametys.plugins.repository.AmetysRepositoryException;
import org.ametys.plugins.repository.data.holder.ModelAwareDataHolder;
import org.ametys.plugins.userdirectory.UserDirectoryHelper;
import org.ametys.plugins.workspaces.members.JCRProjectMember.MemberType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.config.Config;
import org.ametys.runtime.model.View;
import org.ametys.runtime.model.type.DataContext;
import org.ametys.web.WebConstants;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.page.ZoneItem;
import org.ametys.web.usermanagement.UserSignUpConfiguration;

/**
 * Generator used sax project member
 */
public class ProjectMembersGenerator extends ServiceableGenerator
{
    /** The project manager component */
    protected ProjectManager _projectManager;
    
    /** The project member manager */
    protected ProjectMemberManager _projectMemberManager;
    
    /** The user helper */
    protected UserHelper _userHelper;
    
    /** The user directory helper */
    protected UserDirectoryHelper _userDirectoryHelper;
    
    /** The content type extension point */
    protected ContentTypeExtensionPoint _cTypeEP;
    
    /** The group manager */
    protected GroupManager _groupManager;

    /** The sign up configuration */
    protected UserSignUpConfiguration _signupConfig;

    /** The right manager */
    protected RightManager _rightManager;

    @Override
    public void service(ServiceManager serviceManager) throws ServiceException
    {
        _projectManager = (ProjectManager) serviceManager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) serviceManager.lookup(ProjectMemberManager.ROLE);
        _userHelper = (UserHelper) serviceManager.lookup(UserHelper.ROLE);
        _userDirectoryHelper = (UserDirectoryHelper) serviceManager.lookup(UserDirectoryHelper.ROLE);
        _cTypeEP = (ContentTypeExtensionPoint) serviceManager.lookup(ContentTypeExtensionPoint.ROLE);
        _groupManager = (GroupManager) serviceManager.lookup(GroupManager.ROLE);
        _signupConfig = (UserSignUpConfiguration) serviceManager.lookup(UserSignUpConfiguration.ROLE);
        _rightManager = (RightManager) serviceManager.lookup(RightManager.ROLE);
    }

    @Override
    public void generate() throws IOException, SAXException, ProcessingException
    {
        Request request = ObjectModelHelper.getRequest(objectModel);
        
        Page page = (Page) request.getAttribute(WebConstants.REQUEST_ATTR_PAGE);
        String lang = page.getSitemapName();
        
        ZoneItem zoneItem = (ZoneItem) request.getAttribute(WebConstants.REQUEST_ATTR_ZONEITEM);
        ModelAwareDataHolder serviceParameters2018 = zoneItem.getServiceParameters();
        
        long maxNbMembers = serviceParameters2018.getValue("nbMembers", false, -1L);
        boolean expandGroup = serviceParameters2018.getValue("expandGroup", false, false);
        
        contentHandler.startDocument();
        
        List<Project> projects = _projectManager.getProjectsForSite(page.getSite());
        Project project = projects.isEmpty() ? null : projects.get(0);
        
        if (project == null)
        {
            throw new IllegalArgumentException("There is no project for site " + page.getSiteName());
        }
        
        try
        {
            Set<ProjectMember> members = _projectMemberManager.getProjectMembers(project, expandGroup);
            
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("nbMembers", String.valueOf(members.size()));
            
            String memberPageId = _getMembersPageId(project);
            if (memberPageId != null)
            {
                attrs.addCDATAAttribute("memberPageId", memberPageId);
            }
            
            String siteName = Config.getInstance().getValue("workspaces.member.userdirectory.site.name");
            if (StringUtils.isNotEmpty(siteName))
            {
                attrs.addCDATAAttribute("userDirectorySiteName", siteName);
            }
            XMLUtils.startElement(contentHandler, "project", attrs);
            
            UserIdentity[] managers = project.getManagers();
            
            int count = 0;
            Iterator<ProjectMember> it = members.iterator();
            while (it.hasNext() && (maxNbMembers < 1 || count < maxNbMembers))
            {
                ProjectMember member = it.next();
                _saxMember(member, managers, lang);
                count++;
            }
            
            XMLUtils.endElement(contentHandler, "project");
        }
        catch (AmetysRepositoryException e)
        {
            throw new ProcessingException("Unable to get project members", e);
        }
        
        contentHandler.endDocument();
    }
    
    /**
     * Get the id of member page if exists and current user has access
     * @param project The project
     * @return the id of member page. Can be null.
     */
    protected String _getMembersPageId(Project project)
    {
        Set<Page> memberPages = _projectManager.getModulePages(project, MembersWorkspaceModule.MEMBERS_MODULE_ID);
        
        if (!memberPages.isEmpty())
        {
            Page memberPage = memberPages.iterator().next();
            
            if (_rightManager.currentUserHasReadAccess(memberPage))
            {
                return memberPage.getId();
            }
        }
        
        return null;
    }
    /**
     * SAX a member
     * @param member the member
     * @param managers the user identities of project's manager
     * @param lang the current language
     * @throws SAXException if an error occurred while saxing
     * @throws AmetysRepositoryException if an error occurred
     * @throws IOException if failed to sax user content
     */
    protected void _saxMember (ProjectMember member, UserIdentity[] managers, String lang) throws SAXException, AmetysRepositoryException, IOException
    {
        AttributesImpl attrs = new AttributesImpl();
        attrs.addCDATAAttribute("manager", String.valueOf(member.isManager()));
        
        if (member.isManager())
        {
            // Get the position of this manager
            int position = ArrayUtils.indexOf(managers, member.getUser().getIdentity());
            attrs.addCDATAAttribute("managerPosition", String.valueOf(position + 1));
        }
        
        attrs.addCDATAAttribute("type", member.getType().name().toLowerCase());
        
        XMLUtils.startElement(contentHandler, "member", attrs);
        
        XMLUtils.createElement(contentHandler, "title", member.getTitle());
        XMLUtils.createElement(contentHandler, "sortabletitle", member.getSortableTitle());
        String role = member.getRole();
        if (StringUtils.isNotBlank(role))
        {
            XMLUtils.createElement(contentHandler, "role", role);
        }
        
        if (member.getType().equals(MemberType.USER))
        {
            User user = member.getUser();
            _saxUser(user, lang);
        }
        
        XMLUtils.endElement(contentHandler, "member");
    }
    
    /**
     * Sax user
     * @param user the user
     * @param lang the lang
     * @throws SAXException if an error occurs saxing data
     * @throws AmetysRepositoryException if an repository error occurs
     * @throws IOException if an IO exception error occurs
     */
    protected void _saxUser(User user, String lang) throws SAXException, AmetysRepositoryException, IOException
    {
        XMLUtils.startElement(contentHandler, "user");
        _userHelper.saxUser(user, contentHandler, "info");
        Content userContent = _userDirectoryHelper.getUserContent(user.getIdentity(), lang);
        if (userContent != null)
        {
            AttributesImpl contentAttrs = new AttributesImpl();
            contentAttrs.addCDATAAttribute("id", userContent.getId());
            XMLUtils.startElement(contentHandler, "content", contentAttrs);
            ContentType contentType = _cTypeEP.getExtension(UserDirectoryHelper.ABSTRACT_USER_CONTENT_TYPE);
            View view = contentType.getView("abstract");
            userContent.dataToSAX(contentHandler, view, DataContext.newInstance().withLocale(LocaleUtils.toLocale(lang)).withEmptyValues(false));
            XMLUtils.endElement(contentHandler, "content");
        }
        XMLUtils.endElement(contentHandler, "user");
        
    }
}
