/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.members.observers;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.collections4.CollectionUtils;

import org.ametys.cms.ObservationConstants;
import org.ametys.cms.repository.Content;
import org.ametys.core.observation.Event;
import org.ametys.core.observation.Observer;
import org.ametys.plugins.repository.version.VersionableAmetysObject;
import org.ametys.plugins.userdirectory.observation.AbstractContentObserver;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.members.MemberHelper;
import org.ametys.web.repository.page.Page;

/**
 * {@link Observer} to invalidate zone item cache of related members when the skills or keywords have changed 
 *
 */
public class InvalidateZoneItemCacheOnMemberModificationObserver extends AbstractContentObserver
{
    private MemberHelper _memberHelper;

    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _memberHelper = (MemberHelper) smanager.lookup(MemberHelper.ROLE);
    }
    
    public int getPriority()
    {
        // processed just before front-office cache invalidation
        return MAX_PRIORITY + 3500;
    }
    
    public boolean supports(Event event)
    {
        String eventId = event.getId();
        boolean eventSupported =  eventId.equals(ObservationConstants.EVENT_CONTENT_DELETING)  
                || eventId.equals(ObservationConstants.EVENT_CONTENT_UNTAG_LIVE) 
                || eventId.equals(ObservationConstants.EVENT_CONTENT_VALIDATED);
        
        if (eventSupported)
        {
            Content content = (Content) event.getArguments().get(ObservationConstants.ARGS_CONTENT);
            return content != null && _contentTypeHelper.isInstanceOf(content, WorkspacesConstants.MEMBER_CONTENT_TYPE_ID);
        }
        
        return false;
    }

    @Override
    protected void _internalObserve(Event event, Page rootUsersPage, Content userContent)
    {
        try
        {
            if (_needCacheInvalidation(event, userContent))
            {
                Set<Content> contentsToInvalidate = new HashSet<>();
                
                List<Content> relatedMembers = _memberHelper.getRelatedMembers(userContent, Integer.MAX_VALUE);
                contentsToInvalidate.addAll(relatedMembers);
                
                if (userContent instanceof VersionableAmetysObject && !event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING))
                {
                    VersionableAmetysObject versionableContent = (VersionableAmetysObject) userContent;
                    String[] allRevisions = versionableContent.getAllRevisions();
                    if (allRevisions.length > 1)
                    {
                        String lastRevision = allRevisions[allRevisions.length - 2];
                        
                        versionableContent.switchToRevision(lastRevision);
                        try
                        {
                            List<Content> oldRelatedMemmbers = _memberHelper.getRelatedMembers(userContent, Integer.MAX_VALUE);
                            contentsToInvalidate.addAll(oldRelatedMemmbers);
                        }
                        finally
                        {
                            versionableContent.switchToRevision(null);
                        }
                    }
                }
                
                // Invalidate zoneitem cache for new and old related members, for all workspaces
                for (Content content : contentsToInvalidate)
                {
                    _removeZoneItemCache(rootUsersPage, content, null);
                }
            }
            
        }
        catch (Exception e)
        {
            getLogger().error("Unable to get the related members for content '{}'. The zoneitem cache has not been invalidated.", userContent.getId(), e);
        }
        
    }
    
    /**
     * Determines if the zoneitem cache invalidation is needed
     * @param event The event
     * @param userContent the content
     * @return true if zoneitem cache invalidation is needed
     */
    protected boolean _needCacheInvalidation(Event event, Content userContent)
    {
        if (event.getId().equals(ObservationConstants.EVENT_CONTENT_DELETING))
        {
            return true;
        }
        else
        {
            List<String> skills = _memberHelper.getSkills(userContent);
            List<String> keywords = _memberHelper.getKeywords(userContent);
            
            if (userContent instanceof VersionableAmetysObject)
            {
                VersionableAmetysObject versionableContent = (VersionableAmetysObject) userContent;
                String[] allRevisions = versionableContent.getAllRevisions();
                if (allRevisions.length > 1)
                {
                    String lastRevision = allRevisions[allRevisions.length - 2];
                    
                    versionableContent.switchToRevision(lastRevision);
                    try
                    {
                        List<String> oldSkills = _memberHelper.getSkills(userContent);
                        List<String> oldKeywords = _memberHelper.getKeywords(userContent);
                        
                        // Need cache invalidation if the skills or keywords have changed
                        return CollectionUtils.disjunction(skills, oldSkills).size() > 0 || CollectionUtils.disjunction(keywords, oldKeywords).size() > 0;
                    }
                    finally
                    {
                        versionableContent.switchToRevision(null);
                    }
                }
            }
            
            return false;
        }
            
    }

}
