/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.news;

import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.commons.lang3.StringUtils;

import org.ametys.cms.data.Binary;
import org.ametys.cms.repository.Content;
import org.ametys.cms.search.content.ContentSearcherFactory;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.data.holder.ModifiableModelAwareDataHolder;
import org.ametys.plugins.workspaces.AbstractWorkspaceModule;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.WorkspacesConstants;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.util.StatisticColumn;
import org.ametys.plugins.workspaces.util.StatisticsColumnType;
import org.ametys.runtime.i18n.I18nizableText;
import org.ametys.web.repository.page.ModifiablePage;
import org.ametys.web.repository.page.ModifiableZone;
import org.ametys.web.repository.page.ModifiableZoneItem;
import org.ametys.web.repository.page.Page.PageType;
import org.ametys.web.repository.page.ZoneItem.ZoneType;
import org.ametys.web.repository.sitemap.Sitemap;
import org.ametys.web.search.query.SiteQuery;

/**
 * Workspaces module for news
 */
public class NewsWorkspaceModule extends AbstractWorkspaceModule
{
    /** The id of calendar module */
    public static final String NEWS_MODULE_ID = NewsWorkspaceModule.class.getName();
    
    /** The id of news service */
    public static final String NEWS_SERVICE_ID = "org.ametys.web.service.SearchService";
    
    /** Name of news's template */
    public static final String NEWS_PAGE_TEMPLATE = "news";

    /** Search service content types */
    protected static final String[] SEARCH_SERVICE_CONTENT_TYPES = new String[] {WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID};
    
    /** Search service returnables */
    protected static final String[] SEARCH_SERVICE_RETURNABLES = new String[] {"org.ametys.web.frontoffice.search.metamodel.impl.ContentReturnable"};
    
    /** Search service sorts */
    protected static final String[] SEARCH_SERVICE_SORTS = new String[] {"{\"name\":\"ContentReturnable$ContentSearchable$org.ametys.plugins.workspaces.Content.abstractNews$start-date\",\"sort\":\"DESC\"}", "{\"name\":\"ContentReturnable$ContentSearchable$org.ametys.plugins.workspaces.Content.abstractNews$end-date\",\"sort\":\"DESC\"}"};
    
    /** Search service contexts */
    protected static final String[] SEARCH_SERVICE_CONTEXTS = new String[] {"{\"sites\":\"{\\\"context\\\":\\\"CURRENT_SITE\\\",\\\"sites\\\":[]}\",\"search-sitemap-context\":\"{\\\"context\\\":\\\"CURRENT_SITE\\\",\\\"page\\\":null}\",\"context-lang\":\"CURRENT\",\"tags\":[]}"};
    
    /** Search service content view */
    protected static final String SEARCH_SEARCH_CONTENT_VIEW = "portlet";
    
    /** Search service xslt */
    protected static final String SEARCH_SERVICE_XSLT = "pages/services/search/project-news.xsl";
    
    /** Workspaces calendars node name */
    private static final String __WORKSPACES_NEWS_NODE_NAME = "news";

    private static final String __NEWS_NUMBER_HEADER_ID = __WORKSPACES_NEWS_NODE_NAME + "$news_number";

    private ContentSearcherFactory _contentSearcherFactory;

    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _contentSearcherFactory = (ContentSearcherFactory) smanager.lookup(ContentSearcherFactory.ROLE);
    }
    
    public String getId()
    {
        return NEWS_MODULE_ID;
    }

    public int getOrder()
    {
        return ORDER_NEWS;
    }

    public String getModuleName()
    {
        return __WORKSPACES_NEWS_NODE_NAME;
    }

    public Set<String> getAllowedEventTypes()
    {
        return Set.of(ObservationConstants.EVENT_PROJECT_NEWS_PUBLISHED, org.ametys.cms.ObservationConstants.EVENT_CONTENT_COMMENT_VALIDATED);
    }

    @Override
    protected String getModulePageName()
    {
        return "news";
    }

    public I18nizableText getModuleTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_MODULE_NEWS_LABEL");
    }
    public I18nizableText getModuleDescription()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_MODULE_NEWS_DESCRIPTION");
    }

    @Override
    protected I18nizableText getModulePageTitle()
    {
        return new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_WORKSPACE_PAGE_NEWS_TITLE");
    }
    
    @Override
    protected ModifiablePage _createModulePage(Project project, Sitemap sitemap, String name, I18nizableText pageTitle, String skinTemplate)
    {
        if (!sitemap.hasChild(name))
        {
            ModifiablePage page = sitemap.createChild(name, "ametys:defaultPage");
            
            // Title should not be missing, but just in case if the i18n message or the whole catalog does not exists in the requested language
            // to prevent a non-user-friendly error and still generate the project workspace.
            page.setTitle(StringUtils.defaultIfEmpty(_i18nUtils.translate(pageTitle, sitemap.getName()), "Missing title"));
            page.setType(PageType.CONTAINER);
            page.setTemplate(NEWS_PAGE_TEMPLATE);
            page.setSiteName(sitemap.getSiteName());
            page.setSitemapName(sitemap.getName());
            
            sitemap.saveChanges();
            
            return page;
        }
        else
        {
            return null;
        }
    }

    @Override
    protected void initializeModulePage(ModifiablePage modulePage)
    {
        modulePage.untag("SECTION");
        
        ModifiableZone defaultZone;
        if (modulePage.hasZone("default"))
        {
            defaultZone = modulePage.getZone("default");
        }
        else
        {
            defaultZone = modulePage.createZone("default");            
        }
        
        boolean hasService = defaultZone.getZoneItems().stream().anyMatch(zi -> NEWS_SERVICE_ID.equals(zi.getServiceId()));
        
        if (!hasService)
        {
            ModifiableZoneItem defaultZoneItem = defaultZone.addZoneItem();
            defaultZoneItem.setType(ZoneType.SERVICE);
            defaultZoneItem.setServiceId(NEWS_SERVICE_ID);
            
            ModifiableModelAwareDataHolder serviceDataHolder = defaultZoneItem.getServiceParameters();
            serviceDataHolder.setValue("header", null);
            serviceDataHolder.setValue("contentTypes", SEARCH_SERVICE_CONTENT_TYPES);
            serviceDataHolder.setValue("returnables", SEARCH_SERVICE_RETURNABLES);
            serviceDataHolder.setValue("initialSorts", SEARCH_SERVICE_SORTS);
            serviceDataHolder.setValue("contexts", SEARCH_SERVICE_CONTEXTS);
            serviceDataHolder.setValue("resultsPerPage", 12);
            serviceDataHolder.setValue("rightCheckingMode", "exact");
            serviceDataHolder.setValue("resultPlace", "ABOVE_CRITERIA");
            serviceDataHolder.setValue("launchSearchAtStartup", true);
            serviceDataHolder.setValue("rss", false);
            serviceDataHolder.setValue("contentView", SEARCH_SEARCH_CONTENT_VIEW);
            serviceDataHolder.setValue("xslt", SEARCH_SERVICE_XSLT);
        }
    }
    
    @Override
    protected String getModulePageTemplate() 
    {
        return "page";
    }
    
    @Override
    public Map<String, Object> _getInternalStatistics(Project project, boolean isActive)
    { 
        if (isActive)
        {
            Map<String, Object> statistics = new HashMap<>();
            try
            {
                AmetysObjectIterable<Content> results = _contentSearcherFactory.create(WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID)
                        .search(new SiteQuery(project.getName()));
                statistics.put(__NEWS_NUMBER_HEADER_ID, results.getSize());
            }
            catch (Exception e)
            {
                getLogger().error("Error searching news content images in project " + project.getId(), e);
            }
            return statistics;
        }
        else
        {
            return Map.of(__NEWS_NUMBER_HEADER_ID, __SIZE_INACTIVE);
        }
    }
    
    @Override
    public List<StatisticColumn> _getInternalStatisticModel()
    {
        return List.of(new StatisticColumn(__NEWS_NUMBER_HEADER_ID, new I18nizableText("plugin." + _pluginName, "PLUGINS_WORKSPACES_PROJECT_STATISTICS_TOOL_COLUMN_NEWS_NUMBER"))
                .withRenderer("Ametys.plugins.workspaces.project.tool.ProjectsGridHelper.renderElements")
                .withType(StatisticsColumnType.LONG)
                .withGroup(GROUP_HEADER_ELEMENTS_ID));
    }

    @Override
    public long getModuleSize(Project project)
    {
        try
        {
            AmetysObjectIterable<Content> results = _contentSearcherFactory.create(WorkspacesConstants.PROJECT_NEWS_CONTENT_TYPE_ID)
                    .search(new SiteQuery(project.getName()));

            return results.stream()
                .map(content -> content.getValue("illustration/image"))
                .filter(Objects::nonNull)
                .filter(Binary.class::isInstance)
                .map(Binary.class::cast)
                .mapToLong(Binary::getLength)
                .sum();
        }
        catch (Exception e)
        {
            getLogger().error("Error searching news images in project " + project.getId(), e);
            return -1;
        }
    }

    @Override
    protected boolean _showModuleSize()
    {
        return true;
    }

    @Override
    protected boolean _showActivatedStatus()
    {
        return false;
    }
}
