/*
 *  Copyright 2025 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.preferences;

import java.util.Map;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;

import org.ametys.core.ui.Callable;
import org.ametys.core.userpref.UserPreferencesException;
import org.ametys.core.userpref.UserPreferencesManager;
import org.ametys.plugins.workspaces.AbstractWorkspaceDAO;

/**
 * DAO for workspaces view user preferences, such as choosing between cards or list view
 */
public class WorkspaceViewUserPreferencesDAO extends AbstractWorkspaceDAO
{
    /** Avalon Role */
    public static final String ROLE = WorkspaceViewUserPreferencesDAO.class.getName();

    /** the user preferences context for workspaces view */
    public static final String VIEW_USER_PREF_CONTEXT = "/workspaces/view";
    
    /** The user preferences */
    protected UserPreferencesManager _userPrefsManager;

    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _userPrefsManager = (UserPreferencesManager) manager.lookup(UserPreferencesManager.ROLE);
    }

    /**
     * Update the user preferences of current user for a view on given module
     * @param moduleId the module id
     * @param preferences the preferences value
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED) // Preferences are user-specific, so no rights check is needed
    public void setUserPreference(String moduleId, String preferences)
    {
        try
        {
            _userPrefsManager.addUserPreference(_currentUserProvider.getUser(), VIEW_USER_PREF_CONTEXT, Map.of(), moduleId, preferences);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occured while setting the user preferences.", e);
        }
    }

    /**
     * Get the view preference of the current user on given module
     * @param moduleId the module id
     * @return the view preference (true for grid, false for list)
     */
    @Callable (rights = Callable.NO_CHECK_REQUIRED) // Preferences are user-specific, so no rights check is needed
    public String getUserPreference(String moduleId)
    {
        String userPreference = null;
        try
        {
            userPreference = _userPrefsManager.getUserPreferenceAsString(_currentUserProvider.getUser(), VIEW_USER_PREF_CONTEXT, Map.of(), moduleId);
        }
        catch (UserPreferencesException e)
        {
            getLogger().error("An error occured while getting the user preferences.", e);
        }
        return userPreference;
    }
}
