/*
 *  Copyright 2021 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */

package org.ametys.plugins.workspaces.project.notification.schedule;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.time.ZonedDateTime;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.components.ContextHelper;
import org.apache.cocoon.environment.Request;
import org.apache.commons.lang3.StringUtils;
import org.apache.excalibur.source.Source;
import org.apache.excalibur.source.SourceResolver;
import org.apache.excalibur.source.SourceUtil;
import org.quartz.JobExecutionContext;

import org.ametys.core.right.RightManager;
import org.ametys.core.schedule.progression.ContainerProgressionTracker;
import org.ametys.core.user.User;
import org.ametys.core.util.DateUtils;
import org.ametys.core.util.I18nUtils;
import org.ametys.core.util.mail.SendMailHelper;
import org.ametys.plugins.core.impl.schedule.AbstractStaticSchedulable;
import org.ametys.plugins.explorer.resources.ModifiableResourceCollection;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.AmetysObjectResolver;
import org.ametys.plugins.repository.UnknownAmetysObjectException;
import org.ametys.plugins.repository.activities.Activity;
import org.ametys.plugins.repository.activities.ActivityFactory;
import org.ametys.plugins.repository.activities.ActivityHelper;
import org.ametys.plugins.repository.query.expression.AndExpression;
import org.ametys.plugins.repository.query.expression.DateExpression;
import org.ametys.plugins.repository.query.expression.Expression;
import org.ametys.plugins.repository.query.expression.Expression.Operator;
import org.ametys.plugins.repository.query.expression.StringExpression;
import org.ametys.plugins.workspaces.ObservationConstants;
import org.ametys.plugins.workspaces.activities.AbstractWorkspacesActivityType;
import org.ametys.plugins.workspaces.activities.calendars.AbstractCalendarEventActivityType;
import org.ametys.plugins.workspaces.activities.documents.DocumentsActivityType;
import org.ametys.plugins.workspaces.activities.documents.ResourceReferenceElementType.ResourceReference;
import org.ametys.plugins.workspaces.activities.forums.ThreadActivityType;
import org.ametys.plugins.workspaces.activities.minisite.MinisiteActivityType;
import org.ametys.plugins.workspaces.activities.projects.MemberAddedActivityType;
import org.ametys.plugins.workspaces.activities.projects.WebContentActivityType;
import org.ametys.plugins.workspaces.activities.tasks.TasksActivityType;
import org.ametys.plugins.workspaces.members.ProjectMemberManager;
import org.ametys.plugins.workspaces.members.ProjectMemberManager.ProjectMember;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.modules.WorkspaceModule;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper;
import org.ametys.plugins.workspaces.project.notification.preferences.NotificationPreferencesHelper.Frequency;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.runtime.i18n.I18nizable;
import org.ametys.web.WebConstants;
import org.ametys.web.renderingcontext.RenderingContext;
import org.ametys.web.renderingcontext.RenderingContextHandler;
import org.ametys.web.repository.site.Site;
import org.ametys.web.repository.site.SiteManager;

/**
 * Abstract Class to send a mail with the summary of all notification for a time period
 */
public abstract class AbstractSendNotificationSummarySchedulable extends AbstractStaticSchedulable
{
    /** The project manager */
    protected ProjectManager _projectManager;
    /** The notification helper */
    protected NotificationPreferencesHelper _notificationPrefHelper;

    private I18nUtils _i18nUtils;
    private ProjectMemberManager _projectMemberManager;
    private RenderingContextHandler _renderingContextHandler;
    private AmetysObjectResolver _resolver;
    private RightManager _rightManager;
    private SiteManager _siteManager;
    private SourceResolver _srcResolver;


    @Override
    public void service(ServiceManager manager) throws ServiceException
    {
        super.service(manager);
        _i18nUtils = (I18nUtils) manager.lookup(I18nUtils.ROLE);
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
        _projectMemberManager = (ProjectMemberManager) manager.lookup(ProjectMemberManager.ROLE);
        _renderingContextHandler = (RenderingContextHandler) manager.lookup(RenderingContextHandler.ROLE);
        _resolver = (AmetysObjectResolver) manager.lookup(AmetysObjectResolver.ROLE);
        _rightManager = (RightManager) manager.lookup(RightManager.ROLE);
        _siteManager = (SiteManager) manager.lookup(SiteManager.ROLE);
        _srcResolver = (SourceResolver) manager.lookup(SourceResolver.ROLE);
        _notificationPrefHelper = (NotificationPreferencesHelper) manager.lookup(NotificationPreferencesHelper.ROLE);
    }

    @Override
    public void execute(JobExecutionContext context, ContainerProgressionTracker progressionTracker) throws Exception
    {
        List<Map<String, Object>> activitiesInTimeFrame = _getAggregatedActivitiesInTimeFrame();
        if (activitiesInTimeFrame.size() == 0)
        {
            return;
        }
        
        Set<User> userToNotify = _getUserToNotify();
        for (User user : userToNotify)
        {
            Map<String, Map<String, List<Map<String, Object>>>> userActivitiesInTimeFrame = _getActivitiesByProject(user, activitiesInTimeFrame);
            if (userActivitiesInTimeFrame.isEmpty())
            {
                continue;
            }
            
            String lang = StringUtils.defaultIfBlank(user.getLanguage(), _userLanguagesManager.getDefaultLanguage());
            
            String mailSubject = _i18nUtils.translate(getI18nSubject(), lang);
            
            String mailBody = _getMailBody(user, userActivitiesInTimeFrame, lang);
            
            SendMailHelper.newMail()
                .withRecipient(user.getEmail())
                .withSubject(mailSubject)
                .withHTMLBody(mailBody)
                .withInlineCSS(false)
                .withAsync(true)
                .sendMail();
        }
    }

    /**
     * Get all activities that happened in the time frame.
     * Activities are sorted by project
     * @return the activities, sorted by project
     */
    private List<Map<String, Object>> _getAggregatedActivitiesInTimeFrame()
    {
        Expression projectExpr = new StringExpression(AbstractWorkspacesActivityType.PROJECT_NAME, Operator.NE, "");

        ZonedDateTime frameLimit = getTimeFrameLimit();
        Expression dateExpr = new DateExpression(ActivityFactory.DATE, Operator.GE, frameLimit);

        Expression finalExpr = new AndExpression(projectExpr, dateExpr);
        String query = ActivityHelper.getActivityXPathQuery(finalExpr);
        try (AmetysObjectIterable<Activity> activities = _resolver.query(query))
        {
            List<Map<String, Object>> activitiesAsJSON = new ArrayList<>();
            for (Activity activity : activities)
            {
                try
                {
                    activitiesAsJSON.add(activity.toJSONForClient());
                }
                catch (Exception e)
                {
                    getLogger().warn("An error occurred while serializing activity '{}'. The activity will be ignored in the notification summary", activity.getId(), e);
                }
            }

            return _aggregateActivities(activitiesAsJSON);
        }
    }

    private List<Map<String, Object>> _aggregateActivities(List<Map<String, Object>> activities)
    {
        List<Map<String, Object>> aggregatedActivities = new ArrayList<>();
        for (Map<String, Object> activity : activities)
        {

            // We retrieve all the activities that are similar
            List<Map<String, Object>> sameActivities = aggregatedActivities.stream().filter(e -> _isAlreadyPresent(activity, e)).collect(Collectors.toList());
            if (sameActivities.size() == 0)
            {
                // None are find, we just add the new activity
                aggregatedActivities.add(activity);
            }
            else
            {
                try
                {
                    // Find the one we want to store
                    Map<String, Object> finalActivity = activity;
                    ZonedDateTime finalActivityDate = DateUtils.parseZonedDateTime((String) finalActivity.get(ActivityFactory.DATE));
                    boolean multipleAuthor = false;
                    int nbOfOccurrence = 1;
                    // There should never be more than one at a time still
                    for (Map<String, Object> sameActivity : sameActivities)
                    {
                        if (sameActivity.get("nbOfOccurence") != null)
                        {
                            nbOfOccurrence += (int) sameActivity.get("nbOfOccurrence");
                        }
                        else
                        {
                            nbOfOccurrence++;
                        }
                        // If we find similar activity with different author we will have to change the author
                        if (!multipleAuthor && !finalActivity.get(ActivityFactory.AUTHOR).equals(sameActivity.get(ActivityFactory.AUTHOR)))
                        {
                            multipleAuthor = true;
                        }
                        ZonedDateTime activityDate = DateUtils.parseZonedDateTime((String) sameActivity.get(ActivityFactory.DATE));
                        if (activityDate.isAfter(finalActivityDate))
                        {
                            finalActivity = sameActivity;
                            finalActivityDate = activityDate;
                        }
                    }
                    if (multipleAuthor)
                    {
                        finalActivity.remove(ActivityFactory.AUTHOR);
                    }
                    // We removed all the similar activities
                    aggregatedActivities.removeAll(sameActivities);
                    finalActivity.put("nbOfOccurrence", nbOfOccurrence);
                    aggregatedActivities.add(finalActivity);
                }
                catch (Exception e)
                {
                    getLogger().warn("An error occurred while trying to aggregate activity '{}'. The activity will be added as is.", activity.get(Activity.ACTIVITY_ID_KEY), e);
                    aggregatedActivities.add(activity);
                }
            }
        }
        return aggregatedActivities;
    }

    private boolean _isAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        if (!activity.get(AbstractWorkspacesActivityType.PROJECT_NAME).equals(activity2.get(AbstractWorkspacesActivityType.PROJECT_NAME)))
        {
            return false;
        }
        
        if (!activity.get(ActivityFactory.TYPE).equals(activity2.get(ActivityFactory.TYPE)))
        {
            return false;
        }

        String type = (String) activity.get(ActivityFactory.TYPE);
        switch (StringUtils.substringBefore(type, "."))
        {
            case "calendar":
                return _calendarActivityAlreadyPresent(activity, activity2);
            case "member":
                return _memberActivityAlreadyPresent(activity, activity2);
            case "task":
                // Do not merge comment activity. They all contains a different comment.
                if (org.ametys.plugins.workspaces.ObservationConstants.EVENT_TASK_COMMENTED.equals(type))
                {
                    return false;
                }
                return _taskActivityAlreadyPresent(activity, activity2);
            case "resource":
                // Do not merge comment activity. They all contains a different comment.
                if (org.ametys.plugins.workspaces.ObservationConstants.EVENT_RESOURCE_COMMENTED.equals(type))
                {
                    return false;
                }
                return _resourceActivityAlreadyPresent(activity, activity2);
            case "wallcontent":
                return _wallContentActivityAlreadyPresent(activity, activity2);
            case "minisite":
                return _minisiteActivityAlreadyPresent(activity, activity2);
            case "forumthread":
                // Do not merge comment activity. They all contains a different comment.
                if (ObservationConstants.EVENT_THREAD_COMMENTED.equals(type))
                {
                    return false;
                }
                return _forumActivityAlreadyPresent(activity, activity2);
            default:
                return false;
        }
    }

    private boolean _wallContentActivityAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        if (activity.get(WebContentActivityType.CONTENT_ID) != null && activity2.get(WebContentActivityType.CONTENT_ID) != null)
        {
            return activity.get(WebContentActivityType.CONTENT_ID).equals(activity2.get(WebContentActivityType.CONTENT_ID));
        }
        return false;
    }

    private boolean _resourceActivityAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        List<Map<String, Object>> fileList2 = _getFiles(activity2);
        List<Map<String, Object>> fileList = _getFiles(activity);

        // We currently don't have a strategy to merge an activity targeting multiple files
        // The only case where there could be more than one entry in the list is with resource.created events
        // As a resource can only be created once, there should never be a need to merge activity with multiple resources
        if (fileList.size() == 1 && fileList2.size() == 1)
        {
            Object file1 = fileList.get(0).get(ResourceReference.ID);
            Object file2 = fileList2.get(0).get(ResourceReference.ID);
            return file1 != null && file2 != null && file1.equals(file2);
        }
        return false;
    }

    private List<Map<String, Object>> _getFiles(Map<String, Object> activity)
    {
        if (activity.get(DocumentsActivityType.FILE_DATA_NAME) != null)
        {
            @SuppressWarnings("unchecked")
            Map<String, Object> file = (Map<String, Object>) activity.get(DocumentsActivityType.FILE_DATA_NAME);
            return List.of(file);
        }
        else if (activity.get(DocumentsActivityType.FILES_DATA_NAME) != null)
        {
            @SuppressWarnings("unchecked")
            List<Map<String, Object>> files = (List<Map<String, Object>>) activity.get(DocumentsActivityType.FILES_DATA_NAME);
            return files;
        }
        else
        {
            // This case should never happened
            return List.of();
        }
    }

    private boolean _taskActivityAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        if (activity.get(TasksActivityType.TASK_ID) != null && activity2.get(TasksActivityType.TASK_ID) != null)
        {
            return activity.get(TasksActivityType.TASK_ID).equals(activity2.get(TasksActivityType.TASK_ID));
        }
        return false;
    }

    private boolean _memberActivityAlreadyPresent(Map<String, Object> event, Map<String, Object> event2)
    {
        if (event.get(MemberAddedActivityType.MEMBER) != null && event2.get(MemberAddedActivityType.MEMBER) != null)
        {
            return event.get(MemberAddedActivityType.MEMBER).equals(event2.get(MemberAddedActivityType.MEMBER));
        }
        return false;
    }

    private boolean _calendarActivityAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        if (activity.get(AbstractCalendarEventActivityType.CALENDAR_EVENT_ID) != null && activity2.get(AbstractCalendarEventActivityType.CALENDAR_EVENT_ID) != null)
        {
            return activity.get(AbstractCalendarEventActivityType.CALENDAR_EVENT_ID).equals(activity2.get(AbstractCalendarEventActivityType.CALENDAR_EVENT_ID));
        }
        return false;
    }

    private boolean _minisiteActivityAlreadyPresent(Map<String, Object> activity, Map<String, Object> activity2)
    {
        if (activity.get(MinisiteActivityType.PAGE_ID) != null && activity2.get(MinisiteActivityType.PAGE_ID) != null)
        {
            return activity.get(MinisiteActivityType.PAGE_ID).equals(activity2.get(MinisiteActivityType.PAGE_ID));
        }
        return false;
    }

    private boolean _forumActivityAlreadyPresent(Map<String, Object> event, Map<String, Object> event2)
    {
        if (event.get(ThreadActivityType.THREAD_ID) != null && event2.get(ThreadActivityType.THREAD_ID) != null)
        {
            return event.get(ThreadActivityType.THREAD_ID).equals(event2.get(ThreadActivityType.THREAD_ID));
        }
        return false;
    }

    /**
     * Get the earliest activity's date we want to retrieve
     * @return the date after which we want to retrieve activity
     */
    protected abstract ZonedDateTime getTimeFrameLimit();

    /**
     *  Get all user who have this time frame set in there userPrefs
     * @return the list of user to notify
     */
    private Set<User> _getUserToNotify()
    {
        Set<User> users = new HashSet<>();
        try (AmetysObjectIterable<Project> projects = _projectManager.getProjects())
        {
            // Get all members of projects
            for (Project project : projects)
            {
                Set<ProjectMember> projectMembers = _projectMemberManager.getProjectMembers(project, true);
                Set<User> projectUsers = projectMembers.stream().map(ProjectMember::getUser).collect(Collectors.toSet());
                users.addAll(projectUsers);
            }

            return users.stream()
                    // if the user has no mail then no need to compute notification.
                    .filter(user -> StringUtils.isNotEmpty(user.getEmail()))
                    // ensure that the user has this frequency somewhere in his preference
                    .filter(user -> _notificationPrefHelper.hasFrequencyInPreferences(user, getFrequency()))
                    .collect(Collectors.toSet());
        }

    }

    /**
     * Get the notification frequency
     * @return the frequency
     */
    protected abstract Frequency getFrequency();

    private Map<String, Map<String, List<Map<String, Object>>>> _getActivitiesByProject(User user, List<Map<String, Object>> activitiesInTimeFrame)
    {
        Set<String> projectNames = _notificationPrefHelper.getUserProjectsWithFrequency(user, getFrequency());

        Map<String, Map<String, List<Map<String, Object>>>> userActivities = new HashMap<>();
        for (String projectName : projectNames)
        {
            try
            {
                Set<String> allowedType = _getAllowedEventTypes(user, projectName);
                Map<String, List<Map<String, Object>>> userActivitiesByType = _getUserProjectActivities(projectName, allowedType, activitiesInTimeFrame);
                if (!userActivitiesByType.isEmpty())
                {
                    userActivities.put(projectName, userActivitiesByType);
                }
            }
            catch (UnknownAmetysObjectException e)
            {
                // user has a project preference for a non existing project
                // Let's remove it and keep going
                _notificationPrefHelper.deleteProjectNotificationPreferences(user.getIdentity(), projectName);
            }
        }
        return userActivities;
    }

    private Set<String> _getAllowedEventTypes(User user, String projectName)
    {
        Project project = _projectManager.getProject(projectName);
        if (project == null)
        {
            // project does not exist anymore, ignore all events
            return Set.of();
        }

        Set<String> allowedTypes = new HashSet<>();
        for (WorkspaceModule module : _projectManager.getModules(project))
        {
            ModifiableResourceCollection moduleRoot = module.getModuleRoot(project, false);
            if (moduleRoot != null && _rightManager.hasReadAccess(user.getIdentity(), moduleRoot))
            {
                allowedTypes.addAll(module.getAllowedEventTypes());
            }
        }

        return allowedTypes;
    }

    private Map<String, List<Map<String, Object>>> _getUserProjectActivities(String projectName, Set<String> allowedType, List<Map<String, Object>> activitiesInTimeFrame)
    {
        Map<String, List<Map<String, Object>>> userActivitiesByType = new HashMap<>();
        for (Map<String, Object> activityJSON : activitiesInTimeFrame)
        {
            String eventType = (String) activityJSON.get(ActivityFactory.TYPE);
            Object project = activityJSON.get(AbstractWorkspacesActivityType.PROJECT_NAME);
            if (project != null && allowedType.contains(eventType) && project.equals(projectName))
            {
                if (!userActivitiesByType.containsKey(eventType))
                {
                    userActivitiesByType.put(eventType, new ArrayList<>());
                }
                // TODO we should implement here if we send all notification or only those concerning the user
                userActivitiesByType.get(eventType).add(activityJSON);
            }
        }
        return userActivitiesByType;
    }

    /**
     * Get the subject of the mail
     * @return the subject of the mail
     */
    protected abstract I18nizable getI18nSubject();

    private String _getMailBody(User user, Map<String, Map<String, List<Map<String, Object>>>> activities, String lang)
    {

        Source source = null;
        RenderingContext currentContext = _renderingContextHandler.getRenderingContext();
        Request request = ContextHelper.getRequest(_context);

        try
        {
            // Force rendering context.FRONT to resolve URI
            _renderingContextHandler.setRenderingContext(RenderingContext.FRONT);

            request.setAttribute("lang", lang);
            String siteName = _projectManager.getCatalogSiteName();
            Site site = _siteManager.getSite(siteName);

            request.setAttribute("forceAbsoluteUrl", true);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE, site);
            request.setAttribute(WebConstants.REQUEST_ATTR_SITE_NAME, siteName);
            request.setAttribute(WebConstants.REQUEST_ATTR_SKIN_ID, site.getSkinId());

            source = _srcResolver.resolveURI("cocoon://_plugins/workspaces/notification-mail-summary.html",
                                             null,
                                             Map.of("frequency", getFrequency().name(), "activities", activities, "user", user));

            try (InputStream is = source.getInputStream())
            {
                ByteArrayOutputStream bos = new ByteArrayOutputStream();
                SourceUtil.copy(is, bos);

                return bos.toString("UTF-8");
            }
        }
        catch (IOException e)
        {
            throw new RuntimeException("Failed to get mail body for workspaces notifications", e);
        }
        finally
        {
            _renderingContextHandler.setRenderingContext(currentContext);

            if (source != null)
            {
                _srcResolver.release(source);
            }
        }
    }
}
