/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.project.rights.convertor;

import java.util.HashSet;
import java.util.List;
import java.util.Set;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.avalon.framework.service.Serviceable;

import org.ametys.core.right.RightContextConvertor;
import org.ametys.plugins.workspaces.project.ProjectManager;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.web.repository.page.Page;
import org.ametys.web.repository.site.Site;

/**
 * Converts a page of a project to its module root
 * The index page of a project will be additionally converted to the Project
 */
public class PageToModuleRootRightContextConvertor implements RightContextConvertor, Serviceable
{
    private ProjectManager _projectManager;
    
    public void service(ServiceManager manager) throws ServiceException
    {
        _projectManager = (ProjectManager) manager.lookup(ProjectManager.ROLE);
    }
    
    private Project _getProject(Page page)
    {
        Site site = page.getSite();
        List<Project> projectsForSite = _projectManager.getProjectsForSite(site);
        if (projectsForSite.isEmpty())
        {
            return null;
        }
        else
        {
            return projectsForSite.get(0);
        }
    }
    
    public Set<Object> convert(Object object)
    {
        if (object instanceof Page)
        {
            Page page = (Page) object;
            
            Project project = _getProject(page);
            if (project == null)
            {
                return Set.of(); // optimization to avoid to call _projectManager.pageToModuleRoot on a non-project site
            }

            Set<Object> results = new HashSet<>();
            if ("index".equals(page.getPathInSitemap()))
            {
                results.add(project);
            }
            
            results.addAll(_projectManager.pageToModuleRoot(page));
            
            return results;
        }
        else
        {
            return Set.of();
        }
    }
}
