/*
 *  Copyright 2020 Anyware Services
 *
 *  Licensed under the Apache License, Version 2.0 (the "License");
 *  you may not use this file except in compliance with the License.
 *  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 *  Unless required by applicable law or agreed to in writing, software
 *  distributed under the License is distributed on an "AS IS" BASIS,
 *  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 *  See the License for the specific language governing permissions and
 *  limitations under the License.
 */
package org.ametys.plugins.workspaces.search.module;

import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.stream.Collectors;

import org.apache.avalon.framework.service.ServiceException;
import org.apache.avalon.framework.service.ServiceManager;
import org.apache.cocoon.environment.Request;
import org.apache.cocoon.xml.AttributesImpl;
import org.apache.cocoon.xml.XMLUtils;
import org.apache.commons.collections4.IterableUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.solr.client.solrj.util.ClientUtils;

import org.ametys.cms.content.indexing.solr.SolrFieldNames;
import org.ametys.cms.contenttype.ContentTypesHelper;
import org.ametys.cms.search.SearchResult;
import org.ametys.cms.search.SearchResults;
import org.ametys.cms.search.SearchResultsIterable;
import org.ametys.cms.search.SearchResultsIterator;
import org.ametys.cms.search.query.AndQuery;
import org.ametys.cms.search.query.DocumentTypeQuery;
import org.ametys.cms.search.query.FullTextQuery;
import org.ametys.cms.search.query.OrQuery;
import org.ametys.cms.search.query.Query;
import org.ametys.cms.search.query.Query.Operator;
import org.ametys.cms.search.solr.SearcherFactory;
import org.ametys.cms.search.solr.SearcherFactory.Searcher;
import org.ametys.cms.transformation.xslt.ResolveURIComponent;
import org.ametys.core.util.DateUtils;
import org.ametys.plugins.explorer.resources.Resource;
import org.ametys.plugins.repository.AmetysObject;
import org.ametys.plugins.repository.AmetysObjectIterable;
import org.ametys.plugins.repository.EmptyIterable;
import org.ametys.plugins.workspaces.WorkspacesHelper;
import org.ametys.plugins.workspaces.documents.DocumentWorkspaceModule;
import org.ametys.plugins.workspaces.project.objects.Project;
import org.ametys.plugins.workspaces.search.query.ProjectQuery;

/**
 * Generator for posts search module
 */
public class DocumentSearchModuleGenerator extends AbstractSolrSearchModuleGenerator
{
    /** Searcher Factory */
    protected SearcherFactory _searcherFactory;
    /** The content types helper */
    protected ContentTypesHelper _cTypesHelper;
    /** Workspace Helper */
    protected WorkspacesHelper _workspaceHelper;
    
    @Override
    public void service(ServiceManager smanager) throws ServiceException
    {
        super.service(smanager);
        _searcherFactory = (SearcherFactory) manager.lookup(SearcherFactory.ROLE);
        _cTypesHelper = (ContentTypesHelper) manager.lookup(ContentTypesHelper.ROLE);
        _workspaceHelper = (WorkspacesHelper) manager.lookup(WorkspacesHelper.ROLE);
    }
    
    @SuppressWarnings("unchecked")
    @Override
    protected SearchResults< ? extends AmetysObject> getSearchResults(String siteName, String lang, String textfield, Request request, int offset, int limit) throws Exception
    {
        Searcher searcher = _searcherFactory.create();
        
        List<Project> projects = getProjects(request, true);
        
        projects = filterProjectsForModule(projects, DocumentWorkspaceModule.DOCUMENT_MODULE_ID);
        if (projects.isEmpty())
        {
            // No projet availables for this module
            return new EmptySearchResults();
        }
        
        Collection<Query> queries = new HashSet<>();
        
        DocumentTypeQuery documentTypeQuery = new DocumentTypeQuery("projectResource");
        queries.add(documentTypeQuery);
        
        if (StringUtils.isNotBlank(textfield))
        {
            Query fullTextQuery = new FullTextQuery("*" + textfield + "*", lang, Operator.SEARCH);

            Query fileNameQuery = () -> SolrFieldNames.FILENAME + ":(*" + ClientUtils.escapeQueryChars(textfield) + "*)";
            Query searchQuery = new OrQuery(fullTextQuery, fileNameQuery);
            queries.add(searchQuery);
        }
        
        Set<String> projectIds = projects.stream()
                .map(Project::getId)
                .collect(Collectors.toSet());
        queries.add(new ProjectQuery(projectIds));
        
        Query fullQuery = new AndQuery(queries);
        
        return searcher
                .withLimits(offset, limit)
                .withQuery(fullQuery)
                .searchWithFacets();
    }

    @Override
    protected void saxHit(AmetysObject object, String lang) throws Exception
    {
        if (object instanceof Resource)
        {
            Resource file = (Resource) object;
            AttributesImpl attrs = new AttributesImpl();
            attrs.addCDATAAttribute("id", file.getId());
            XMLUtils.startElement(contentHandler, "hit", attrs);
            
            XMLUtils.createElement(contentHandler, "title", file.getName());
            XMLUtils.createElement(contentHandler, "uri", ResolveURIComponent.resolve("project-resource", file.getId(), false, true));
            XMLUtils.createElement(contentHandler, "length", String.valueOf(file.getLength()));
            XMLUtils.createElement(contentHandler, "mimeType", file.getMimeType());
            XMLUtils.createElement(contentHandler, "fileType", _workspaceHelper.getFileType(file).name().toLowerCase());
            
            Date creationDate = file.getCreationDate();
            XMLUtils.createElement(contentHandler, "creationDate", DateUtils.dateToString(creationDate));
            
            Date lastModified = file.getLastModified();
            XMLUtils.createElement(contentHandler, "lastModified", DateUtils.dateToString(lastModified));
            
            Project project = getProject(object);
            saxProject(project);
            XMLUtils.endElement(contentHandler, "hit");
        }
    }
    
    class EmptySearchResults<A extends AmetysObject> implements SearchResults<A>
    {
        public SearchResultsIterable<SearchResult<A>> getResults()
        {
            return new EmptySearchResultsIterable<>();
        }

        public AmetysObjectIterable<A> getObjects()
        {
            return new EmptyIterable<>();
        }

        public Iterable<String> getObjectIds()
        {
            return IterableUtils.emptyIterable();
        }

        public Map<String, Map<String, Integer>> getFacetResults()
        {
            return Collections.EMPTY_MAP;
        }

        public long getTotalCount()
        {
            return 0;
        }

        public float getMaxScore()
        {
            return 0;
        }

        public Optional<Map<String, Object>> getDebugMap()
        {
            return Optional.empty();
        }
        
    }
    
    class EmptySearchResultsIterable<A extends AmetysObject> implements SearchResultsIterable<SearchResult<A>>
    {

        public long getSize()
        {
            return 0;
        }

        @SuppressWarnings("unchecked")
        public SearchResultsIterator<SearchResult<A>> iterator()
        {
            return new EmptySearchResultsIterator();
        }
    }
    
    class EmptySearchResultsIterator<A extends AmetysObject> implements SearchResultsIterator<SearchResult<A>>
    {
        EmptySearchResultsIterator()
        {
            // Empty
        }
        
        @Override
        public boolean hasNext()
        {
            return false;
        }
        
        @Override
        public SearchResult<A> next()
        {
            throw new UnsupportedOperationException();
        }
        
        @Override
        public void skip(long skipNum)
        {
            throw new UnsupportedOperationException();
        }
    }
}
